/*
 *  Barebone WiFi webserver for the ESP8266/32
 *  
 *  Copyright 2019, 2024 Enoch Hwang
 */

// The two tokens ESP32 and ESP8266 are automatically defined
// when the respective board is selected from the menu Tools/Board

#define SERVERPORT 80 // default server port for http
#ifdef ESP32
  #include <WiFi.h>
  #include <WebServer.h>
  WebServer server(SERVERPORT);  // create a webserver object to listen on server port
#else
  #include <ESP8266WiFi.h>
  #include <ESP8266WebServer.h>
  ESP8266WebServer server(SERVERPORT);  // create a webserver object to listen on server port
#endif
#include "time.h"

const char ssid[] = "your ssid";           // change this to your ssid
const char password[] = "your password";   // change this to your password
char devicename[] = "ESP8266/32 barebone";

#ifdef ESP32
  #define LEDON    1
  #define LEDOFF   0  // turn off led with a 0 for ESP32
  #define LEDBLINK 2
  #define statusLed 2
#else
  #define LEDON    0
  #define LEDOFF   1  // turn off led with a 1 for ESP8266
  #define LEDBLINK 2
  #define statusLed 2
#endif
int ledStatus;

// Flashes the led without using the delay() function.
// The period of each cycle is flashDelay milliseconds
// flashSpeed determines the speed of each flash
// flashTimes determines the number of times to flash in a cycle
// This version needs to be called inside a loop in order to flash
// In order to notice a pause between each cycle
// flashDelay should be greater than 2*flashSpeed*flashTimes
//  e.g.   flashActivityLed(1000, 100, 1);
void flashActivityLed(int flashDelay=3000, int flashSpeed=150, int flashTimes=2, bool resetTime=false) {
  static unsigned long nextLedActivity = 0;
  static int t = 0;
  if (ledStatus != LEDBLINK) {
    digitalWrite(statusLed, ledStatus); // turn on or off LED
    return;  // return if ledStatus is 0=off or 1=on
  }
  // blink the LED
  if (resetTime) {
    nextLedActivity = 0; // reset next flash time
    digitalWrite(statusLed, LEDOFF); // turn off LED
  }
  if ((millis() > nextLedActivity)) {
    if (t == 0) {
      // digitalWrite(statusLed, LOW); // turn on LED
      digitalWrite(statusLed, LEDON); // turn on LED
    } else {
      digitalWrite(statusLed, !digitalRead(statusLed)); // toggle LED
    }
    if (t < flashTimes*2-1) {
      nextLedActivity = millis() + flashSpeed;
      t++;
    } else {
      nextLedActivity = millis() + flashDelay - (flashSpeed*2*flashTimes);
      t = 0;
    }
  }
}

// Returns a human readable time string
// Friday 2/28/2020 11:44:53 am  ddd mm/dd/yyyy hh:mm:ss am
// Time is stored either as time_t or struct tm
// time_t is referred to as machine time or unix time.
// It is generally implemented as the number of seconds elapsed since 
// 00:00 hours, Jan 1, 1970 UTC.
// struct tm is referred to as human readable time which is a structure 
// containing a calendar date and time broken down into its components
String getTime() {
  String wd[] = {"Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"};
  String s = "";
  struct tm timeinfo;

  // get current date and time;
  #ifdef ESP32
    if (getLocalTime(&timeinfo)) {  // only for ESP32
  #else // only for ESP8266
    time_t t = time(nullptr);
    timeinfo = *localtime(&t);
    if (t) {
  #endif

    //s = s + wd[timeinfo.tm_wday].substring(0,3) + " ";
    s = s + wd[timeinfo.tm_wday] + " ";
    s = s + String(timeinfo.tm_mon+1) + "/";
    s = s + String(timeinfo.tm_mday) + "/";
    s = s + String(timeinfo.tm_year+1900) + " ";

    int h = timeinfo.tm_hour;
    String ampm = "";  
    if (h > 12) {
      h = h - 12;
      ampm = "pm";
    } else {
      ampm = "am";
    }
    if (h < 10) s = s + "0";
    s = s + String(h) + ":";
    if (timeinfo.tm_min < 10) s = s + "0";
    s = s + String(timeinfo.tm_min) + ":";
    if (timeinfo.tm_sec < 10) s = s + "0";
    s = s + String(timeinfo.tm_sec) + " " + ampm + " ";
  } else {
    s = "No time available yet";
  }
  return s;
}

// HTML root webpage
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("LED")) {
    if (String(server.arg("LED")) == "OFF") {
      digitalWrite(statusLed, LEDOFF);  // turn off LED
      ledStatus = LEDOFF;
      Serial.println("turn off LED");
    } else if (String(server.arg("LED")) == "ON") {
      digitalWrite(statusLed, LEDON);   // turn on LED
      ledStatus = LEDON;
      Serial.println("turn on LED");
    } else if (String(server.arg("LED")) == "BLINK") {
      ledStatus = LEDBLINK;                // blink LED
      Serial.println("blink LED");
    }
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<title>ESP</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>ESP8266/32 WiFi Demo Web Server</b></font><br>";
  msg += "This is an ESP8266/32 wifi demo webserver that has been running for ";
  msg += millis()/1000;
  msg += " seconds.";
  msg += "<br>You can control the built-in LED with this webpage.";
  msg += "<br>" + getTime();
  msg += "</center>";
  msg += "<hr><br>";

  msg += "Current led status is ";
  switch(ledStatus) {
    case LEDOFF: 
      msg += "<b>OFF</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=ON\"><button style=\"font-size:16pt\">ON</button></a>";
      break;
    case LEDON: 
      msg += "<b>ON</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=BLINK\"><button style=\"font-size:16pt\">BLINK</button></a>";
      break;
    default: 
      msg += "<b>BLINK</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a>";
      break;
  }

  // another way to create the buttons
  msg += "<br><br>Another way to do it:";
  msg += "<form style=\"font-size:12pt\">";
  msg += "Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn off led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn on led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"BLINK\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to blink led";
  msg += "</form>";
  
  msg += "<br><hr>IP address: ";
  msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
//  msg += "<br>Wifi setup and info: ";
//  msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  msg += "<center><a href=\"http://robotsforfun.com\"><img width=\"100\" src=\"http://robotsforfun.com/images/Robo.png\" alt=\"Robo the robot\"></a></center>";

  msg += "</body>";
  msg += "</html>";
  // end of webpage file

  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}

void setup() {
  Serial.begin(115200);
  Serial.println();
  pinMode(statusLed, OUTPUT);
  ledStatus = LEDBLINK;

  // config time zone PST with automatic daylight changing time for the internet time
  // https://www.gnu.org/software/libc/manual/html_node/TZ-Variable.html
  configTzTime("PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", "pool.ntp.org", "time.nist.gov");

  // optional to set device name
  WiFi.hostname(devicename);
  // optional to set ip address
  // IPAddress ip(192, 168, 1, 99);
  // IPAddress gateway(192, 168, 1, 1);  // the ip address and the gateway have to be the same
  // IPAddress dns(192, 168, 1, 1);
  // IPAddress subnet(255, 255, 255, 0);
  // WiFi.config(ip, gateway, subnet, dns);          // must have dns, otherwise cannot connect to NTP server

  Serial.print("Connecting WiFi to [");
  Serial.print(ssid);
  Serial.print("] using password [");
  Serial.print(password);
  Serial.println("]");
  WiFi.begin(ssid, password); // start WiFi with password
  //  WiFi.begin(ssid); // start WiFi without password
  unsigned long startTime = millis();
  while ((WiFi.status() != WL_CONNECTED) && (millis() - startTime < 10000)) {  // 10 seconds timeout
    digitalWrite(statusLed, !digitalRead(statusLed));
    delay(125);
    Serial.print(".");
  }
  Serial.println();
  if (WiFi.status() == WL_CONNECTED) {    
    digitalWrite(statusLed, LEDON);
    IPAddress IP = WiFi.localIP();
    WiFi.mode(WIFI_STA);      // use just WiFi station mode
    Serial.println("WiFi connected");
    Serial.print("  IP address: ");
    Serial.println(IP);
    Serial.print("  Port:       ");
    Serial.println(SERVERPORT);
    Serial.print("  Gateway:    ");
    Serial.println(WiFi.gatewayIP().toString());
    Serial.print("  Hostname:   ");
    #ifdef ESP32
      Serial.println(WiFi.getHostname()); // for ESP32
    #else
      Serial.println(WiFi.hostname());    // for ESP8266
    #endif
    long rssi = WiFi.RSSI();  // receive signal strength
    Serial.print("  Signal strength RSSI: ");
    Serial.println(rssi);

    // wifi is connected to local access point
    // but it is possible that the access point is not connected to the internet
    // so see if we can reach the internet by querying a DNS server
    IPAddress ipaddress;
    if (WiFi.hostByName("www.google.com", ipaddress) == 1) { // Connect to DNS server to get IP address
      Serial.println("Internet connected");
      server.on("/", indexHTML);  // set root webpage
      server.begin();             // start web server
      Serial.println("Web server started");
      Serial.print("  Browse to ");
      Serial.print(IP);
      Serial.println(" to access webpage");
      // wait for time valid
      // when the clock is not set time(0) returns the value 28803
      // which is for Thu Jan  1 08:00:03 1970
      // so a valid time is greater than 28803
      Serial.println(getTime());
      while (time(0) < 30000) {
        server.handleClient();
        yield();  // must have this to give time to handle wifi background stuff
      }
    } else {
      Serial.println("Internet not connected");
    }
  } else {
    Serial.println("WiFi is not connected");
    while (true) {
      flashActivityLed(2000,150,1);
      server.handleClient();
      yield();  // must have this to give time to handle wifi background stuff
    }
  }
}

void loop() {
  static time_t prevDisplay;
  if (time(0) != prevDisplay) { // display the time only if the second has changed
    prevDisplay = time(0);
    Serial.println(getTime());  // having outputs in the loop slows down web response a lot
  }
  
  flashActivityLed();
  server.handleClient();  // handle web client request
  //yield();  // ESP8266 need this to give time to handle wifi background stuff
              // ESP32 don't need to give time to handle wifi background stuff because ESP32 has 2 CPU cores
  //delay(1); // don't need this for faster web response
}
