/* //////////////////////////////////////////////////////////
 * A Distance/Object detection webserver using the ESP8266-12
 * and the HC-SR04 ultrasonic sensor
 *
 * Connections:
 * ESP8266-12       Ultrasonic Sensor       5V to 3.3V          5V Power supply
 *                                          power regulator
 *    4                   Trig              
 *    5                   Echo
 *                        Vcc               5V Vin              5V
 *    3.3V                                  3.3V Out
 *    Gnd                 Gnd               Gnd                 Gnd
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address. For example
 *     192.168.1.99            for the main webpage
 *   or
 *     192.168.1.99/setup      for the WiFi setup page
 * 
 * or
 * 
 * Method 2 for after the program is up and running.
 * Connect your WiFi to this access point with the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or the one you provided, then browse to the IP address
 *     1.2.3.4                  for the main webpage
 *   or
 *     1.2.3.4/setup            for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to
 * use method 2 above to connect your WiFi to the access point
 * and then browse to 1.2.3.4/setup to setup the ssid and password.
 * Alternatively, you can specify the ssid and password to connect to
 * in the setupWiFi(ssid, password) function call.
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the built-in led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
ADC_MODE(ADC_VCC);

//////////////////////////////////////////////////////////////
//// status LED stuff
#define statusLed 2     // for ESP8266-12

//////////////////////////////////////////////////////////////
//// EEPROM stuff
#include <EEPROM.h>
#define EEPROM_ADDRESS_distanceMode EEPROM_START_OF_DATA
#define EEPROM_ADDRESS_detectionCount EEPROM_ADDRESS_distanceMode+sizeof(int)

//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include <DateTime_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// WiFi stuff
//#define SERVERPORT 1040        // set server port. Default is 80
//#define USE_DEFAULT_INDEXHTML  // define this to use the default indexHTML function
#include <WiFi_RobotsForFun.h>

//////////////////////////////////////////////////////////////
// Push Notification and PHP stuff
// PHP file on RobotsForFun.com
// robotsforfun.com/iot/push_notification.php
#include "PushNotifications_EMails.h"
bool pushNotificationEnabled = false;

//////////////////////////////////////////////////////////////
// Event history stuff
#define MAXRECORD 30
char history_event[MAXRECORD];
int history_distance[MAXRECORD];
int history_month[MAXRECORD];
int history_day[MAXRECORD];
int history_hour[MAXRECORD];
int history_minute[MAXRECORD];
int historyCount = 0;

String eventString(char event, int d=0) {
    if (event == 'A')
      return "Alarm ON";
    if (event == 'a')
      return "Alarm OFF";
    if (event == 'D')
      return "Object detected at " + String(d) + " cm";
    if (event == 'd')
      return "No object";
    if (event == 'S')
      return "Sensor stuck";
    if (event == 'R')
      return "System ready";
}

void rememberEvent(char event, int d=0) {
  if (historyCount >= MAXRECORD) {
    // shift all records down
    for (int i=1; i<MAXRECORD; i++) {
      history_event[i-1] = history_event[i];
      history_distance[i-1] = history_distance[i];
      history_month[i-1] = history_month[i];
      history_day[i-1] = history_day[i];
      history_hour[i-1] = history_hour[i];
      history_minute[i-1] = history_minute[i];
    }
    historyCount--;
  }
  history_event[historyCount] = event;
  history_distance[historyCount] = d;
  history_month[historyCount] = month();
  history_day[historyCount] = day();
  history_hour[historyCount] = hour();
  history_minute[historyCount] = minute();
  historyCount++;
}

String getEvents() {
  String s = "";
  String t;
  s += "  <select name=\"folder\" style=\"width:345px; font-size:12pt\">";
  for (int i=historyCount-1; i>=0; i--) {
    s += "<option style=\"font-size:12pt\">";
    if (i < 10) {
      s += "&nbsp;&nbsp;";
    }
    s += "&nbsp;&nbsp;&nbsp;&nbsp;";    
    s += String(history_month[i]) + "/";
    s += String(history_day[i]);
    s += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";

    if (history_hour[i] < 13)
      t = String(history_hour[i]);
    else
      t = String(history_hour[i]-12);
    if (t.length() == 1)
      t = "0" + t;
    s += t;

    if (history_minute[i] < 10)
      t = "0";
    else
      t = "";
    t += String(history_minute[i]);
    s += ":" + t;

    if (history_hour[i] < 12)
      s += " am";
    else    
      s += " pm";

    s += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
    s += eventString(history_event[i], history_distance[i]);
    s += "&nbsp;&nbsp;</option>";  
  }
  s += "</select>";
  return s;
}

//////////////////////////////////////////////////////////////
//// Ultrasonic stuff
#define trigPin 4
#define echoPin 5
#define DISTANCE 1
#define DETECTION 0
boolean distanceMode = DETECTION;
int distanceThreshold = 0;
String lastDetected = "";
int detectionCount = 0;

// this function calculates and returns the distance
// as obtained from the ultrasonic sensor
int distance() {
  // STEP 1
  // The next five lines will send out one ultrasound pulse
  digitalWrite(trigPin, LOW);
  delayMicroseconds(2);
  digitalWrite(trigPin, HIGH);
  delayMicroseconds(10);
  digitalWrite(trigPin, LOW);

  // STEP 2
  // The pulseIn command starts the timing and wait for the echo
  // when it hears the echo it will return the time in microseconds
  int duration = pulseIn(echoPin, HIGH);

  // STEP 3
  // Calculate the distance (in cm) based on the time from STEP 2 and the speed of sound
  // the calculated distance is returned by the function
  return duration/58.2;  // in cm
  //return duration/148;// in inches
} // end of function

// find the statistical mode of the numbers in A
// count is the number of numbers in A
int mode(int A[], int count) {
  int mostcount, currentcount;
  int most, current;
  most = A[0];
  mostcount = 1;
  for(int pos = 0; pos < count; pos++) {
    current = A[pos];
    currentcount = 0;
    for(int inner = pos + 1; inner < count; inner++) {
      if (A[inner] == current) {
        currentcount++;
      }
    }
    if(currentcount > mostcount) {
      most = current;
      mostcount = currentcount;
    }
    // If we have less array slices left than the current
    // maximum count, then there is no room left to find
    // a bigger count.  We have finished early and we can
    // go home.
    if(count - pos < mostcount) {
      break;
    }
  }
  return most;
}

void setThreshold() {
  Serial.println("Setting threshold...");
  int d[10];
  distanceThreshold = 0;
  for (int i=0; i<10; i++) {
    d[i] = distance();
    Serial.println(d[i]);
    if (d[i] <= 0) {  // throw away if < 0
      i--;
      continue;
    }
    digitalWrite(statusLed, HIGH);
    delay(20);
    digitalWrite(statusLed, LOW);
    delay(230);
    Serial.println(d[i]);
  }
  distanceThreshold = mode(d,10)-3; // find mode of the 10 numbers
  if (distanceThreshold < 2) distanceThreshold = 2;
  Serial.print("Threshold set at ");
  Serial.print(distanceThreshold);
  Serial.println(" cm");
}

void detection() {
  static int d = 0;
  static bool objectDetected = false;

  d = distance();

  if (distanceMode == DISTANCE) {
    // measure distance to object
    if ((d > 0) && (d < distanceThreshold)) {
      ledStatus = LEDON;
      digitalWrite(statusLed, !digitalRead(statusLed));
      Serial.println(d);
      nonblockingDelay(50+d*5); // blinking speed depends on the distance
    } else {
      d = 0;
      ledStatus = LEDBLINK;
      flashActivityLed();
    }
//    nonblockingDelay(50+d*4);  // blinking speed depends on the distance
    
  } else {
    // detect object
    if ((d > 0) && (d < distanceThreshold)) {
      if (!objectDetected) {
        objectDetected = true;
        ledStatus = LEDON;
        digitalWrite(statusLed, LOW);   // turn on led
        detectionCount++;
        EEPROM.put(EEPROM_ADDRESS_detectionCount, detectionCount);
        EEPROM.commit();
        rememberEvent('D', d);
        lastDetected = String(d) + " cm away on " + formattedTime() + " count=" + String(detectionCount);
        Serial.print("Object detected at ");
        Serial.println(lastDetected);
      }
    } else {
      if (objectDetected) {
        objectDetected = false;
        // ledStatus = LEDBLINK;
        EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
        rememberEvent('d');
        digitalWrite(statusLed, HIGH);  // turn off led
        Serial.print("No object detected on ");
        Serial.println(formattedTime());
      }
    }
    nonblockingDelay(50);  // must have this short delay, otherwise ultrasound gives errors
  }
}

//////////////////////////////////////////////////////////////
//// HTML webpage
// this routine is called when a client browse to the root IP address
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("mode")) {
    if (server.arg("mode").equals("distance")) {
      if (distanceMode != DISTANCE) {
        Serial.println("Distance mode is on");    
        distanceMode = DISTANCE;
        EEPROM.put(EEPROM_ADDRESS_distanceMode, distanceMode);
        EEPROM.commit();
      }
    } else if (server.arg("mode").equals("detection")) {
      if (distanceMode != DETECTION) {
        Serial.println("Detection mode is on");
        Serial.print("  It will detect objects less than ");
        Serial.print(distanceThreshold);
        Serial.println(" cm away");
        distanceMode = DETECTION;
        EEPROM.put(EEPROM_ADDRESS_distanceMode, distanceMode);
        EEPROM.commit();
      }
    }
  } else if (server.hasArg("pushNotificationEnabled")) {
    if (server.arg("pushNotificationEnabled").equals("ON")) {
      pushNotificationEnabled = true;
    } else {
      pushNotificationEnabled = false;
    }
  } else if (server.hasArg("RESETCOUNT")) {
    Serial.println("Reseting count to 0");
    lastDetected = "";
    detectionCount = 0;
    historyCount = 0;
    EEPROM.put(EEPROM_ADDRESS_detectionCount, 0); // reset alarm count
    EEPROM.commit();
  } else if (server.hasArg("SETTHRESHOLD")) {
    Serial.println("Reseting the distance threshold");
    setThreshold();    
  }


  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<title>Ultrasonic Webserver</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Distance/Object Sensor Webserver</b></font><br>";
  msg += "This is an ESP8266 demo website that tells you if there's an object in front or the distance to an object in front using an ultrasonic distance sensor.";
  msg += "<br>" + formattedTime();
  msg += "</center>";
  msg += "<hr>";

  msg += "<b>Detection Activities:</b>";
  msg += "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Day&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Time&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Event<br>";
  msg += getEvents();
  msg += "<br>";

  msg += "<br>Detection count: " + String(detectionCount);
  msg += "<br>Object detection and distance measurement threshold is set at ";
  msg += String(distanceThreshold);
  msg += " cm.";

  // get distance
  long d = distance();
  if (distanceMode == DISTANCE) {
    if (d < distanceThreshold) {
      msg += "<br>Distance to the nearest object in front is ";
      msg += String(d);
      msg += " cm.";
    } else {
      msg += "<br>Distance to the nearest object is greater than the distance measurement threshold.";
    }
  } else {
    msg += "<br>Current status: ";
    if ((d > 0) && (d < distanceThreshold)) {
      // object detected
      msg += "object detected at ";
      msg += String(d);
      msg += " cm away.";
    } else {
      msg += "no object detected.";      
    }
    msg += "<br>Last time objected detected at ";
    msg += lastDetected;
  }

  msg += "<br><br><b>Distance/Detect Mode select:</b>";
  msg += "<form style=\"font-size:14pt\">";
  if (distanceMode == DISTANCE) {
    msg += "<input type=\"radio\" name=\"mode\" value=\"distance\" style=\"height:40px;width:80px;font-size:16pt\" checked> Distance<br>";
    msg += "<input type=\"radio\" name=\"mode\" value=\"detection\" style=\"height:40px;width:80px;font-size:16pt\"> Detect object<br>";
  } else {
    msg += "<input type=\"radio\" name=\"mode\" value=\"distance\" style=\"height:40px;width:80px;font-size:16pt\"> Distance<br>";
    msg += "<input type=\"radio\" name=\"mode\" value=\"detection\"  style=\"height:40px;width:80px;font-size:16pt\" checked> Detect object<br>";    
  }
  msg += "<input type=\"submit\" value=\"Submit\" style=\"font-size:16pt\">";
  msg += "</form>";

  msg += "<br><br><a href=\"/\?RESETCOUNT=ON\"><button style=\"font-size:16pt\">Reset Count</button></a>";
  msg += "&nbsp;<a href=\"/\?SETTHRESHOLD=ON\"><button style=\"font-size:16pt\">Calibrate Threshold</button></a>";


  msg += "<br><br>Turn push notifications&nbsp;";
  if (pushNotificationEnabled) {
    msg += "<a href=\"/?pushNotificationEnabled=OFF\"><button style=\"font-size:14pt\">OFF</button></a>";
  } else {
    msg += "<a href=\"/?pushNotificationEnabled=ON\"><button style=\"font-size:14pt\">ON</button></a>";
  }
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";
  // end of webpage file

  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}

void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  pinMode(trigPin, OUTPUT);
  pinMode(echoPin, INPUT);
  
  //// Ultrasonic stuff
  setThreshold();

  //// set clock time zone to PST with automatic hourly updates and daylight savings time adjustments
  setClock(PST);
  
  //// WiFi stuff
  char ssid[] = "your ssid";           // change this to your ssid
  char password[] = "your password";   // change this to your password
  char devicename[] = "Ultrasonic Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  IPAddress staticIP(192, 168, 1, 99); // use this if you want a static IP address
  setupAP(devicename);  // setup access point
  setupWiFi(devicename, staticIP, true);
  //setupWiFi(devicename, ssid, password, staticIP, true);

  // wait for clock to sync with server
  waitForTimeValid();

  
  // update systemStartTime and systemStartCount. This is for information purposes only
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
  EEPROM.get(EEPROM_ADDRESS_distanceMode, distanceMode);
  if ((distanceMode != DISTANCE) && (distanceMode != DETECTION)) distanceMode = DETECTION;

  if (distanceMode == DISTANCE) {
    Serial.println("Distance mode is on");
  } else {
    // detect object mode; need to first callibrate value with no object
    Serial.println("Detection mode is on");
    Serial.print("  It will detect objects less than ");
    Serial.print(distanceThreshold);
    Serial.println(" cm away");
  }
}


void loop() {
  detection();
  
  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();     // show system is alive
  server.handleClient();  // handle web client request
  //yield();  // yield() is called automatically at the end of loop() must have this to give time to handle wifi background stuff
}
