/* //////////////////////////////////////////////////////////
 * A web client using the ESP8266
 * that executes a php file on a server.
 * This is acting like a web browser that browses to a php file on a web server
 * It will also pass whatever GET parameters to the php file.
 * For this demo, the php file accesses a mySQL database.
 *
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or to the one you provided, then browse to the IP address
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to go to
 * 1.2.3.4/setup to setup the ssid and password
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
  #define __VCCVOLTAGE__
  ADC_MODE(ADC_VCC);


//////////////////////////////////////////////////////////////
//// status LED stuff
//#define statusLed 1     // for ESP8266-01
#define statusLed 2     // for ESP8266-12
bool statusLedOn = false;

// Flashes the led without using the delay() function.
// The period of each cycle is flashDelay milliseconds
// flashSpeed determines the speed of each flash
// flashTimes determines the number of times to flash in a cycle
// This version needs to be called inside a loop in order to flash
// In order to notice a pause between each cycle
// flashDelay should be greater than 2*flashSpeed*flashTimes
//  e.g.   flashActivityLed(1000, 100, 1);
void flashActivityLed(int flashDelay=3000, int flashSpeed=150, int flashTimes=2) {
  static unsigned long nextLedActivity = 0;
  static int t = 0;
  if (statusLedOn) return;
  if ((millis() > nextLedActivity)) {
    if (t == 0) {
      digitalWrite(statusLed, LOW); // turn on LED
    } else {
      digitalWrite(statusLed, !digitalRead(statusLed)); // toggle LED
    }
    if (t < flashTimes*2-1) {
      nextLedActivity = millis() + flashSpeed;
      t++;
    } else {
      nextLedActivity = millis() + flashDelay - (flashSpeed*2*flashTimes);
      t = 0;
    }
  }
}


//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include "DateTime_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// WiFi stuff
#define INDEXHTML     // need this to define your own index.html page below
#include "WiFi_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// mySQL database on RobotsForFun.com stuff
//// host: robotsforfun.com
//// executes the php file /iot/SQLAPWeb.php
//// the php file accesses the:
////   database name: robotsforfun
////   table name: iotcontrol

const char* host = "robotsforfun.com";
const int httpPort = 80;
const String phpFile = "/iot/SQLAPWeb.php?";

WiFiClient SQLclient;
byte mac_address[6];
String mac;
boolean SQLconnected = false; // whether the SQL database server is connected

boolean SQLconnect() {
  // connect to SQL database server
  if (!SQLclient.connect(host, httpPort)) {
    Serial.println("Cannot connect to SQL server");
    SQLclient.stop();
    return false;
  } else {
    Serial.println("Connected to SQL server");
    return true;
  }
}

#define LISTALL 0
#define SELECT 1
#define INSERT 2
#define UPDATE 3
#define DELETE 4
String SQLsend(int operation = LISTALL, String description = "", String value = "0") {
  if (!SQLconnect()) {
    return "";
  }
  // Create a URL for the request
  String url = phpFile;
  if (operation == LISTALL) {
    url += "operation=LISTALL";
  } else if (operation == SELECT) {
    url += "operation=SELECT";
    url += "&mac="+mac;
  } else if (operation == INSERT) {
    url += "operation=INSERT";
    url += "&mac="+mac;
//    url += "&description=Insert+record";
//    url += "&description=Insert%20record";  // cannot have spaces
    url += "&description="+description;
    url += "&value="+value;
  } else if (operation == UPDATE) {
    url += "operation=UPDATE";
    url += "&mac="+mac;
//    url += "&description=Updated%20record";   // cannot have spaces
    url += "&description="+description;
    url += "&value="+value;
  } else if (operation == DELETE) {
    url += "operation=DELETE";
    url += "&mac="+mac;
  }
  
  Serial.print("URL=");
  Serial.println(url);
  
  // Send the URL request to the server
  SQLclient.print("GET " + url + " HTTP/1.1\r\n" +
               "Host: " + host + "\r\n" + 
               "Connection: close\r\n\r\n");
  int timeout = millis() + 5000;
  while (SQLclient.available() == 0) {
    if (millis() > timeout) {
      Serial.println(">>> Client Timeout !");
      SQLconnected = false;
      SQLclient.stop();
      return "";
    }
  }

  // Read all the lines of the reply from server and print them to Serial
  String line = "";
  while(SQLclient.available()){
    line += SQLclient.readStringUntil('\r');
  }
  // trim off unwanted stuff from the line before printing out
  line = line.substring(line.indexOf("<br>"));
  line.remove(line.length()-4); // remove last unwanted 4 characters from line
  Serial.print(line);
  
  Serial.println("closing connection");
  SQLconnected = true;
  SQLclient.stop();
  return line;
}


//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  
  msg += "<title>ESP8266 Demo</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>mySQL Client Web Server</b></font><br>";
  msg += "This is an ESP8266 demo webserver that allows you to access a mySQL database.";
  msg += "</center>";
  msg += "<hr><br>";

  msg += "  <form style=\"font-size:16pt\">";
  // create the input text fields
  msg += "<br>  Description: <input type=\"text\" name=\"DESCRIPTION\" maxlength=\"30\" style=\"font-size:12pt\">";
  msg += "<br>  Value: <input type=\"text\" name=\"VALUE\" maxlength=\"4\" style=\"font-size:12pt\">";
  msg += "<br><br>";
  // create the buttons
  msg += "  <input type=\"submit\" name=\"SQL\" value=\"LISTALL\" style=\"font-size:16pt\" >";
  msg += "  <input type=\"submit\" name=\"SQL\" value=\"SELECT\" style=\"font-size:16pt\" >";
  msg += "  <input type=\"submit\" name=\"SQL\" value=\"INSERT\" style=\"font-size:16pt\" >";
  msg += "  <input type=\"submit\" name=\"SQL\" value=\"UPDATE\" style=\"font-size:16pt\" >";
  msg += "  <input type=\"submit\" name=\"SQL\" value=\"DELETE\" style=\"font-size:16pt\" >";
  msg += "  </form>";

  msg += "<hr>";  

  // get url arguments and process command
  if (server.args() == 3) { // make sure there are three arguments
    String description = "";
    String value = "";
    if (server.argName(0).equals("DESCRIPTION")) {  // make sure argument 0's name is "DESCRIPTION"
      description = server.arg(0);
      description.replace(" ","%20"); // cannot have spaces
    }
    if (server.argName(1).equals("VALUE")) {  // make sure argument 1's name is "VALUE"
      value = server.arg(1);
      value.replace(" ","%20"); // cannot have spaces
    }
    
    if (server.argName(2).equals("SQL")) {  // make sure argument 2's name is "SQL"
      if (server.arg(2).equals("LISTALL")) {
        Serial.println("LISTALL");
        if (WifiConnected()) {
          msg += SQLsend(LISTALL);
        }
      } else if (server.arg(2).equals("SELECT")) {
        Serial.println("SELECT");
        if (WifiConnected()) {
          msg += SQLsend(SELECT);
        }
      } else if (server.arg(2).equals("INSERT")) {
        Serial.println("INSERT");
        if (WifiConnected()) {
          msg += SQLsend(INSERT,description,value);
        }
      } else if (server.arg(2).equals("UPDATE")) {
        Serial.println("UPDATE");
        if (WifiConnected()) {
          msg += SQLsend(UPDATE,description,value);
        }
      } else if (server.arg(2).equals("DELETE")) {
        Serial.println("DELETE");
        if (WifiConnected()) {
          msg += SQLsend(DELETE);
        }
      }
      
    }  // if (server.argName(0).equals("SQL"))
  } // if (server.args() == 4)


  msg += "<br><hr>Signal strength: ";
  msg += signalStrength();
  msg += "%";
  msg += "<br>Vcc voltage: ";
  int p = (ESP.getVcc() + 5) / 10;
  msg += String(p/100.0);
  msg += "V";
  msg += "<br>IP address: ";
  msg += WiFi.localIP().toString();
  msg += "<br>Hostname: ";
  msg += WiFi.hostname();

  msg += "<br>Setup Wifi: ";
  String m = "";
  if (WifiConnected()) {
    m += WiFi.localIP()[0];
    m += ".";
    m += WiFi.localIP()[1];
    m += ".";
    m += WiFi.localIP()[2];
    m += ".";
    m += WiFi.localIP()[3];
  } else {
    m += "1.2.3.4";
  }
  msg += "<a href=\"http://" + m + "/setup\">" + m + "/setup</a>";
  msg += "<br>Started on: ";
  msg += systemStartTime;

  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send( 200, "text/html", msg);
  
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);

  setupAP();  // setup access point using default ssid "WiFi setup ESP8266-xxxx" and no password
  char devicename[] = "ESP8266 Demo Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  setupWiFi(devicename);  // use DHCP
  
  setClock(PST);

  // get MAC address (just for unique ID, otherwise, no need for SQL operations)
  WiFi.macAddress(mac_address);
  mac = String(mac_address[5],HEX) + ':' + String(mac_address[4],HEX) + ':' + String(mac_address[3],HEX) + ':' + String(mac_address[2],HEX) + ':' + String(mac_address[1],HEX) + ':' + String(mac_address[0],HEX);

  if (WifiConnected()) {
    SQLsend(LISTALL);
  }

}


void loop() {

  // turn off AP after 10 minutes
  static bool APon = true;
  if (APon && (millis() > 600000)) {
    turnOffAP();
    APon = false;
  }
  flashActivityLed();
  checkWifiConnection();
  syncClock(); // need to call this frequently to sync the clock
  server.handleClient();
  yield();  // yield time for the webserver to handle the clients
}
