/* //////////////////////////////////////////////////////////
 * A Motion sensor webserver using the ESP8266-12
 * and the AM312 mini motion sensor
 *
 * Connections:
 * ESP8266-12     AM312 motion sensor       5V to 3.3V        5V Power supply
 *                                          power regulator
 *    5                 Out (this works)
 *    13                Out (this also works)              
 *                      Vcc               5V Vin              5V
 *    3.3V                                3.3V Out
 *    Gnd               Gnd               Gnd                 Gnd
 * 
 * 
 * ESP8266-01     AM312 motion sensor       5V to 3.3V        5V Power supply
 *    3                 Out
 * 
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address. For example
 *     192.168.1.99            for the main webpage
 *   or
 *     192.168.1.99/setup      for the WiFi setup page
 * 
 * or
 * 
 * Method 2 for after the program is up and running.
 * Connect your WiFi to this access point with the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or the one you provided, then browse to the IP address
 *     1.2.3.4                  for the main webpage
 *   or
 *     1.2.3.4/setup            for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to
 * use method 2 above to connect your WiFi to the access point
 * and then browse to 1.2.3.4/setup to setup the ssid and password.
 * Alternatively, you can specify the ssid and password to connect to
 * in the setupWiFi(ssid, password) function call.
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the built-in led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//#define ESP8266_01  // define this if using the ESP8266-01

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
ADC_MODE(ADC_VCC);

//////////////////////////////////////////////////////////////
//// status LED stuff
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01
#else
  #define statusLed 2     // for ESP8266-12 and ESP32
#endif

//////////////////////////////////////////////////////////////
//// EEPROM stuff
#include <EEPROM.h>
#define EEPROM_ADDRESS_detectionCount EEPROM_START_OF_DATA

//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include <DateTime_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// WiFi stuff
//#define SERVERPORT 1040        // set server port. Default is 80
#include <WiFi_RobotsForFun.h>

//////////////////////////////////////////////////////////////
// Push Notification and PHP stuff
// PHP file on RobotsForFun.com
// robotsforfun.com/iot/push_notification.php
#include "PushNotifications_EMails.h"
bool pushNotificationEnabled = false;

//////////////////////////////////////////////////////////////
// Event history stuff
#define MAXRECORD 100
char history_event[MAXRECORD];
int history_month[MAXRECORD];
int history_day[MAXRECORD];
int history_hour[MAXRECORD];
int history_minute[MAXRECORD];
int historyCount = 0;

String eventString(char event) {
    if (event == 'A')
      return "Alarm ON";
    if (event == 'a')
      return "Alarm OFF";
    if (event == 'M')
      return "Motion detected";
    if (event == 'm')
      return "No motion";
    if (event == 'S')
      return "Sensor stuck";
    if (event == 'R')
      return "System ready";
}

void rememberEvent(char event) {
  if (historyCount >= MAXRECORD) {
    // shift all records down
    for (int i=1; i<MAXRECORD; i++) {
      history_event[i-1] = history_event[i];
      history_month[i-1] = history_month[i];
      history_day[i-1] = history_day[i];
      history_hour[i-1] = history_hour[i];
      history_minute[i-1] = history_minute[i];
    }
    historyCount--;
  }
  history_event[historyCount] = event;
  history_month[historyCount] = month();
  history_day[historyCount] = day();
  history_hour[historyCount] = hour();
  history_minute[historyCount] = minute();
  historyCount++;
}

String getEvents() {
  String s = "";
  String t;
  s += "  <select name=\"folder\" style=\"width:345px; font-size:12pt\">";
  for (int i=historyCount-1; i>=0; i--) {
    s += "<option style=\"font-size:12pt\">";
    if (i < 10) {
      s += "&nbsp;&nbsp;";
    }
    s += "&nbsp;&nbsp;&nbsp;&nbsp;";    
    s += String(history_month[i]) + "/";
    s += String(history_day[i]);
    s += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";

    if (history_hour[i] < 13)
      t = String(history_hour[i]);
    else
      t = String(history_hour[i]-12);
    if (t.length() == 1)
      t = "0" + t;
    s += t;

    if (history_minute[i] < 10)
      t = "0";
    else
      t = "";
    t += String(history_minute[i]);
    s += ":" + t;

    if (history_hour[i] < 12)
      s += " am";
    else    
      s += " pm";

    s += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
    s += eventString(history_event[i]);
    s += "&nbsp;&nbsp;</option>";  
  }
  s += "</select>";
  return s;
}

//////////////////////////////////////////////////////////////
//// motion sensor AM312 stuff
#ifdef ESP8266_01
  #define sensorPin 3    // for ESP8266-01
#else
//  #define sensorPin 5    // for ESP8266-12
  #define sensorPin 13    // for ESP8266-12
#endif
int detectionCount = 0;
String turnOnMsg="";
String turnOffMsg="";

void checkMotion() {
  static bool motionDetected = false;
  // active high
  if (digitalRead(sensorPin) == HIGH) {
    // motion detected
    if (!motionDetected) {
      motionDetected = true;
      ledStatus = LEDON;
      digitalWrite(statusLed, LOW);   // turn on led
      detectionCount++;
      EEPROM.put(EEPROM_ADDRESS_detectionCount, detectionCount);
      EEPROM.commit();
      rememberEvent('M');
      turnOnMsg = "Motion detected on " + formattedTime() + " count=" + String(detectionCount);
      Serial.println(turnOnMsg);
    }
  } else {
    // motion not detected
    if (motionDetected) {
      motionDetected = false;
      // ledStatus = LEDBLINK;
      EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
      rememberEvent('m');
      digitalWrite(statusLed, HIGH);  // turn off led      
      turnOffMsg = "          ended on " + formattedTime();
      Serial.println(turnOffMsg);
    }
  }
}


//////////////////////////////////////////////////////////////
//// HTML webpage
// this routine is called when a client browse to the root IP address
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("pushNotificationEnabled")) {
    if (server.arg("pushNotificationEnabled").equals("ON")) {
      pushNotificationEnabled = true;
    } else {
      pushNotificationEnabled = false;
    }
  } else if (server.hasArg("RESETCOUNT")) {
    Serial.println("Reseting count to 0");
    detectionCount = 0;
    historyCount = 0;
    turnOnMsg = "";
    turnOffMsg = "";
    EEPROM.put(EEPROM_ADDRESS_detectionCount, 0); // reset alarm count
    EEPROM.commit();
  }


  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<title>Motion Webserver</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Motion Sensor Webserver</b></font><br>";
  msg += "This is an ESP8266 demo webserver that tells you whether there is motion or not.";
  msg += "<br>" + formattedTime();
  msg += "</center>";
  msg += "<hr>";

  msg += "<b>Motion Activities:</b>";
  msg += "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Day&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Time&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Event<br>";
  msg += getEvents();
  msg += "<br>";

  msg += "<br>";
  msg += turnOnMsg;
  msg += "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
  msg += turnOffMsg;

  msg += "<br><br><a href=\"/\?RESETCOUNT=ON\"><button style=\"font-size:16pt\">Reset Count</button></a>";

  msg += "<br><br>Turn push notifications&nbsp;";
  if (pushNotificationEnabled) {
    msg += "<a href=\"/?pushNotificationEnabled=OFF\"><button style=\"font-size:14pt\">OFF</button></a>";
  } else {
    msg += "<a href=\"/?pushNotificationEnabled=ON\"><button style=\"font-size:14pt\">ON</button></a>";
  }
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";
  // end of webpage file

  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}

void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  pinMode(sensorPin, INPUT);

  //// set clock time zone to PST with automatic hourly updates and daylight savings time adjustments
  setClock(PST);
  
  //// WiFi stuff
  char ssid[] = "your ssid";           // change this to your ssid
  char password[] = "your password";   // change this to your password
  char devicename[] = "Motion Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  IPAddress staticIP(192, 168, 1, 99); // use this if you want a static IP address
  setupAP(devicename);  // setup access point
  setupWiFi(devicename, staticIP, true);
  //setupWiFi(devicename, ssid, password, staticIP, true);

  // wait for clock to sync with server
  waitForTimeValid();

  // update systemStartTime and systemStartCount. This is for information purposes only
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
}


void loop() {
  checkMotion();
  
  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();     // show system is alive
  server.handleClient();  // handle web client request
  //yield();  // yield() is called automatically at the end of loop() must have this to give time to handle wifi background stuff
}
