/* //////////////////////////////////////////////////////////
 * A Web Server using the ESP8266-12
 * that allows you to control a 2x16 LCD display.
 * and show the date and time
 *
 * Connect RS on the LCD to pin D5 on the ESP8266-12
 * Connect E on the LCD to pin D4 on the ESP8266-12
 * Connect D4 on the LCD to pin D16 on the ESP8266-12
 * Connect D5 on the LCD to pin D14 on the ESP8266-12
 * Connect D6 on the LCD to pin D12 on the ESP8266-12
 * Connect D7 on the LCD to pin D13 on the ESP8266-12
 * Connect R/W on the LCD to GND
 * Connect Vo on the LCD to 1K resistor then to GND 
 * Connect A on the LCD to 5V 
 * Connect K on the LCD to 150 resistor then to GND 
 * Connect VSS on the LCD to GND 
 * Connect VDD on the LCD to 5V
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or to the one you provided, then browse to the IP address
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to go to
 * 1.2.3.4/setup to setup the ssid and password
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
  ADC_MODE(ADC_VCC);

//////////////////////////////////////////////////////////////
//// EEPROM stuff
#include <EEPROM.h>
#define EEPROM_ADDRESS_clockOn EEPROM_START_OF_DATA

//////////////////////////////////////////////////////////////
//// status LED stuff
#define statusLed 2     // for ESP8266-12

//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include <DateTime_RobotsForFun.h>
boolean clockOn = false;  // flag to display clock
boolean autoClockOn = false;  // flag to automatically display clock on startup

//////////////////////////////////////////////////////////////
//// WiFi stuff
#include <WiFi_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// LCD stuff
#include <LiquidCrystal.h>
#define RS 5
#define E 4
#define D4 16
#define D5 14
#define D6 12
#define D7 13

LiquidCrystal lcd(RS, E, D4, D5, D6, D7);

void displayClock() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("  ");
  lcd.print(hour12());
  lcd.print(":");
  int m = minute();
  if (m < 10) lcd.print("0");
  lcd.print(m);
  lcd.print(":");
  int s = second();
  if (s < 10) lcd.print("0");
  lcd.print(s);
  lcd.print(" ");
  if (isPM())
    lcd.print(" pm");
  else    
    lcd.print(" am");

  static const char *w[] = {"Sun","Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
  lcd.setCursor(0, 1);
  lcd.print(" ");
  lcd.print(w[weekday()]);
  lcd.print(" ");
  lcd.print(month());
  lcd.print("/");
  lcd.print(day());
  lcd.print("/");
  lcd.print(year());
}

void displayMessage(String s1, String s2) {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print(s1);
  lcd.setCursor(0, 1);
  lcd.print(s2);
}

//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  // get url arguments and process the command
  // do it here to see the changes reflect in the webpage
  // should not have long time delays otherwise the webpage will be very slow to load
  if (server.hasArg("AUTOCLOCK")) {
    if (String(server.arg("AUTOCLOCK")) == "ON") {
      autoClockOn = true;
    } else if (String(server.arg("AUTOCLOCK")) == "OFF") {
      autoClockOn = false;
    }
    EEPROM.put(EEPROM_ADDRESS_clockOn, autoClockOn);
    EEPROM.commit();
  }

  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  
  msg += "<title>LCD Clock</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>LCD Clock/Message Display</b></font><br>";
  msg += "  This is an ESP8266 demo webserver that allows you to control a 2x16 LCD display.";
  msg += "<br>" + formattedTime();
  msg += "</center>";
  msg += "<hr>";
  
  // create text input
  msg += "<b>Message</b><br>";
  msg += "Type in two lines:<br>";
  msg += "<form style=\"font-size:16pt\">";
  msg += "<input type=\"text\" name=\"STRING1\" maxlength=\"16\" style=\"width:200px;font-size:14pt\" >";
  msg += "<br>";
  msg += "<input type=\"text\" name=\"STRING2\" maxlength=\"16\" style=\"width:200px;font-size:14pt\" >";
  msg += "<br>";
  msg += "<input type=\"submit\" style=\"font-size:16pt\">";
  msg += "</form>";
  msg += "<br><hr>";

  msg += "<b>Automatically Sync Clock with NTP Server</b><br>";
  msg += "<a href=\"/?SYNC=ON\"><button style=\"font-size:16pt\">Sync Clock</button></a>";
  msg += "<br><br>";
  
  msg += "<b>Manually Set Clock</b><br>";
  msg += "<form style=\"font-size:12pt\">";
  msg += "&nbsp;Year: <input type=\"number\" name=\"YR\" min=\"1990\" max=\"2035\" maxlength=\"4\" style=\"width:60px;font-size:12pt\" >";
  msg += "&nbsp;Month: <input type=\"number\" name=\"MO\" min=\"1\" max=\"12\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Day: <input type=\"number\" name=\"DA\" min=\"1\" max=\"31\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "<br>Hour: <input type=\"number\" name=\"HR\" min=\"0\" max=\"23\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Minute: <input type=\"number\" name=\"MI\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;Second: <input type=\"number\" name=\"SE\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "<br><br><input type=\"submit\" value=\"Set clock\" style=\"font-size:16pt\">";
  msg += "</form>";

  msg += "<br><br><a href=\"/\?CLOCK=ON\"><button style=\"font-size:16pt\">Display clock</button></a>";
  msg += "<br>Turn auto clock start&nbsp;";
  if (autoClockOn) {
    msg += "<a href=\"/\?AUTOCLOCK=OFF\"><button style=\"font-size:16pt\">Off</button></a>";
  } else {
    msg += "<a href=\"/\?AUTOCLOCK=ON\"><button style=\"font-size:16pt\">On</button></a>";
  }
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send (200, "text/html", msg);

  
  // get url arguments and process the command that DO have time delays
  if (server.hasArg("STRING1")) {
    clockOn = false;
    lcd.clear();
    lcd.setCursor(0, 0);  // set cursor to column 0, line 0
    lcd.print(server.arg("STRING1"));
    lcd.setCursor(0, 1);  // set cursor to column 0, line 1
    lcd.print(server.arg("STRING2"));
    Serial.println(server.arg("STRING1"));
    Serial.println(server.arg("STRING2"));
    
  } else if (server.hasArg("CLOCK") && (server.arg("CLOCK").equals("ON"))) {
    clockOn = true;
  } else if (server.hasArg("YR")) {
    // set the clock
    int yr, mo, da, hr, mi, se;
    yr = server.arg("YR").toInt();
    mo = server.arg("MO").toInt();
    da = server.arg("DA").toInt();
    hr = server.arg("HR").toInt();
    mi = server.arg("MI").toInt();
    se = server.arg("SE").toInt();
    if (yr == 0) yr = year();
    if (mo == 0) mo = month();
    if (da == 0) da = day();
    if (hr == 0) hr = hour();
    if (mi == 0) mi = minute();
    setClock(yr, mo, da, hr, mi, se);
    clockOn = true;
  } else if (server.hasArg("SYNC")) {
    setClock(PST);
  }
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  
  //// LCD stuff
  lcd.begin(16, 2);  // set number of columns and rows
  lcd.clear();
  lcd.setCursor(0, 0);  // set cursor to column 0, line 0
  lcd.print("Connecting to");
  lcd.setCursor(0, 1);  // set cursor to column 0, line 1
  loadWifiCredentials();
  lcd.print(_ssid);

  //// set clock time zone to PST
  setClock(PST);

  //// WiFi stuff
  //char ssid[] = "your ssid";           // change this to your ssid
  //char password[] = "your password";   // change this to your password
  char devicename[] = "LCD clock"; // optional. change to whatever you want. Use "ping ESP8266" to test
  setupAP(devicename);  // setup access point using default ssid "WiFi setup ESP8266-xxxx" and no password
  IPAddress staticIP(192, 168, 1, 99);     // for the LCD clock
  bool blocking = false;
  // if (!setupWiFi(devicename, staticIP, blocking)) {
  if (!setupWiFi(devicename, blocking)) {   // use DHCP
    lcd.clear();
    lcd.setCursor(0, 0);  // set cursor to column 0, line 0
    if (wifiIsConnected) {
      lcd.print("WiFi only");
      lcd.setCursor(0, 1);  // set cursor to column 0, line 1
      lcd.print("Connect to AP");
    } else {
      lcd.print("Failed. Connect");
      lcd.setCursor(0, 1);  // set cursor to column 0, line 1
      lcd.print("to AP to setup");
    }
  } else {
    // connected to wifi
    lcd.clear();
    lcd.setCursor(0, 0);  // set cursor to column 0, line 0
    lcd.print(_ssid);
    lcd.setCursor(0, 1);  // set cursor to column 0, line 1
    lcd.print(WiFi.localIP());  // print IP address on LCD
    digitalWrite(statusLed, HIGH);  // turn off led
    
    // wait for clock to sync with server
    waitForTimeValid();
  }


  // update systemStartTime and systemStartCount
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
  EEPROM.get(EEPROM_ADDRESS_clockOn, autoClockOn);

  if (autoClockOn) {
    // block for 30 seconds to display the IP address
    Serial.println("auto clock display");
    lcd.setCursor(15, 1);  // set cursor to column 15, line 1
    lcd.print(">");
    unsigned long timeout = millis() + 30000; // 30 seconds
    while((millis() < timeout) && !clockOn) { // !clockOn is for early termination if webpage user clicks Display clock
      flashActivityLed();
      server.handleClient();
      yield();  // must have this to give time to handle wifi background stuff
    }
    clockOn = true; // to automatically display clock on startup
  }
}

void loop() {
  static time_t prevDisplay;
  if (clockOn) {
    if (now() != prevDisplay) { //update the display only if time has changed
      prevDisplay = now();
      displayClock();
      Serial.println(formattedTime());
    }
  }
  
  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();     // show system is alive
  server.handleClient();  // handle web client request
}
