/* //////////////////////////////////////////////////////////
 * A Basic Web Server using the ESP8266
 * to set and display the date and time
 *
// Reference for struct tm as defined in <time.h>
// http://www.cplusplus.com/reference/ctime/
// http://www.cplusplus.com/reference/ctime/tm/
// only works with ESP8266 v2.4.0 or higher
// select from the menu Tools | Boards | Board Manager
// then install the ESP8266 v2.4.0 or higher
//
// Reference for configTime
// https://github.com/espressif/arduino-esp32/blob/f41beb92bf4434f52209bd15655c1b2f2447447f/cores/esp32/esp32-hal-time.c
// https://github.com/esp8266/Arduino/blob/9913e5210779d2f3c4197760d6813270dbba6232/cores/esp8266/time.c
// https://github.com/esp8266/Arduino/blob/61cd8d83859524db0066a647de3de3f6a0039bb2/libraries/esp8266/examples/NTP-TZ-DST/NTP-TZ-DST.ino
//
// Note that if the clock is not automatically set initially but just manually set
// then the localtime() and gmtime() are both the same.
// However, if the clock is first automatically set then the localtime() is
// for the given timezone whereas the gmtime() is for UTC.
 *
 *
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address. For example
 *     192.168.1.99            for the main webpage
 *   or
 *     192.168.1.99/setup      for the WiFi setup page
 * 
 * or
 * 
 * Method 2 for after the program is up and running.
 * Connect your WiFi to this access point with the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or the one you provided, then browse to the IP address
 *     1.2.3.4                  for the main webpage
 *   or
 *     1.2.3.4/setup            for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to
 * use method 2 above to connect your WiFi to the access point
 * and then browse to 1.2.3.4/setup to setup the ssid and password.
 * Alternatively, you can specify the ssid and password to connect to
 * in the setupWiFi(ssid, password) function call.
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the built-in led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//#define ESP8266_01  // define this if using the ESP8266-01

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
#ifndef ESP32
  ADC_MODE(ADC_VCC);
#endif

//////////////////////////////////////////////////////////////
//// status LED stuff
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01
#else
  #define statusLed 2     // for ESP8266-12
#endif

//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include <DateTime_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// WiFi stuff
//#define SERVERPORT 1040        // set server port. Default is 80
#include <WiFi_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// HTML webpage
// this routine is called when a client browse to the root IP address
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("YR")) {
    // set the clock
    int yr, mo, da, hr, mi, se;
    yr = server.arg("YR").toInt();
    mo = server.arg("MO").toInt();
    da = server.arg("DA").toInt();
    hr = server.arg("HR").toInt();
    mi = server.arg("MI").toInt();
    se = server.arg("SE").toInt();
    if (yr == 0) yr = year();
    if (mo == 0) mo = month();
    if (da == 0) da = day();
    if (hr == 0) hr = hour();
    if (mi == 0) mi = minute();
    setClock(yr, mo, da, hr, mi, se);
  } else if (server.hasArg("SYNC")) {
    setClock(PST);
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
//  msg += "<meta http-equiv=\"refresh\" content=\"60\">"; // auto refresh page every 1 second
  msg += "<title>Date Time</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Date Time Web Server</b></font><br>";
  msg += formattedTime();
  msg += "</center>";
  msg += "<hr>";

  msg += "<b>Automatically Sync Clock with NTP Server</b><br>";
  msg += "<a href=\"/?SYNC=ON\"><button style=\"font-size:16pt\">Sync Clock</button></a>";
  msg += "<br><br>";
  
  msg += "<b>Manually Set Clock</b><br>";
  msg += "<form style=\"font-size:12pt\">";
  msg += "&nbsp;Year: <input type=\"number\" name=\"YR\" min=\"1990\" max=\"2035\" maxlength=\"4\" style=\"width:60px;font-size:12pt\" >";
  msg += "&nbsp;Month: <input type=\"number\" name=\"MO\" min=\"1\" max=\"12\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Day: <input type=\"number\" name=\"DA\" min=\"1\" max=\"31\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "<br>Hour: <input type=\"number\" name=\"HR\" min=\"0\" max=\"23\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Minute: <input type=\"number\" name=\"MI\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "&nbsp;Second: <input type=\"number\" name=\"SE\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
  msg += "<br><br><input type=\"submit\" value=\"Set clock\" style=\"font-size:16pt\">";
  msg += "</form>";
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";
  // end of webpage file

  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client
}

void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);

  //// set clock time zone to PST with automatic hourly updates and daylight savings time adjustments
  setClock(PST);
  
  //// WiFi stuff
  char ssid[] = "your ssid";           // change this to your ssid
  char password[] = "your password";   // change this to your password
  char devicename[] = "DateTime Webserver";
  IPAddress staticIP(192, 168, 1, 99); // use this if you want a static IP address
  setupAP(devicename);  // setup access point with the devicename as the ssid and with no password
  setupWiFi(devicename, true);
  //setupWiFi(devicename, staticIP, true);
  //setupWiFi(devicename, ssid, password, staticIP, true);

  // wait for clock to sync with server
  waitForTimeValid();
  
  // update systemStartTime and systemStartCount. This is for information purposes only
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
}


void loop() {
  static time_t prevDisplay;
  if (time(nullptr) != prevDisplay) { //update the display only if time has changed
    prevDisplay = now();
    Serial.println(formattedTime());
  }
  
  if (hour24() == 18 && minute() == 21 && second() == 0) {
    Serial.println("It's time");
  }

  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();     // show system is alive
  server.handleClient();  // handle web client request
  //yield();  // yield() is called automatically at the end of loop() must have this to give time to handle wifi background stuff
}
