/*
 *  Barebone Access Point (AP) Example for the ESP8266-12
 *  
 *  Copyright 2019 Enoch Hwang
 */
#include <ESP8266WiFi.h>
#include <ESP8266WebServer.h>
#define SERVERPORT 80 // default server port for http
ESP8266WebServer server(SERVERPORT);  // create a webserver object to listen on server port
char ssid[] = "ESP8266 AP";
char password[] = "12345678";

#define LEDON    0
#define LEDOFF   1  // turn off led with a 1 for ESP8266
#define LEDBLINK 2
#define statusLed 2
int ledStatus;

// Flashes the led without using the delay() function.
// The period of each cycle is flashDelay milliseconds
// flashSpeed determines the speed of each flash
// flashTimes determines the number of times to flash in a cycle
// This version needs to be called inside a loop in order to flash
// In order to notice a pause between each cycle
// flashDelay should be greater than 2*flashSpeed*flashTimes
//  e.g.   flashActivityLed(1000, 100, 1);
void flashActivityLed(int flashDelay=3000, int flashSpeed=150, int flashTimes=2, bool resetTime=false) {
  static unsigned long nextLedActivity = 0;
  static int t = 0;
  if (ledStatus != LEDBLINK) return;  // return if ledStatus is 0=off or 1=on
  if (resetTime) {
    nextLedActivity = 0; // reset next flash time
    digitalWrite(statusLed, LEDOFF); // turn off LED
  }
  if ((millis() > nextLedActivity)) {
    if (t == 0) {
      // digitalWrite(statusLed, LOW); // turn on LED
      digitalWrite(statusLed, LEDON); // turn on LED
    } else {
      digitalWrite(statusLed, !digitalRead(statusLed)); // toggle LED
    }
    if (t < flashTimes*2-1) {
      nextLedActivity = millis() + flashSpeed;
      t++;
    } else {
      nextLedActivity = millis() + flashDelay - (flashSpeed*2*flashTimes);
      t = 0;
    }
  }
}

// HTML root webpage
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("LED")) {
    if (String(server.arg("LED")) == "OFF") {
      digitalWrite(statusLed, LEDOFF);  // turn off LED
      ledStatus = LEDOFF;
      Serial.println("turn off LED");
    } else if (String(server.arg("LED")) == "ON") {
      digitalWrite(statusLed, LEDON);   // turn on LED
      ledStatus = LEDON;
      Serial.println("turn on LED");
    } else if (String(server.arg("LED")) == "BLINK") {
      ledStatus = LEDBLINK;                // blink LED
      Serial.println("blink LED");
    }
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<title>ESP8266 Demo</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>ESP8266 AP Demo Web Server</b></font><br>";
  msg += "This is an ESP8266 demo webserver that has been running for ";
  msg += millis()/1000;
  msg += " seconds.";
  msg += "<br>You can control the built-in LED with this webpage.";
  msg += "</center>";
  msg += "<hr><br>";

  msg += "Current led status is ";
  switch(ledStatus) {
    case LEDOFF: 
      msg += "<b>OFF</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=ON\"><button style=\"font-size:16pt\">ON</button></a>";
      break;
    case LEDON: 
      msg += "<b>ON</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=BLINK\"><button style=\"font-size:16pt\">BLINK</button></a>";
      break;
    default: 
      msg += "<b>BLINK</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a>";
      break;
  }

  // another way to create the buttons
  msg += "<br><br>Another way to do it:";
  msg += "<form style=\"font-size:12pt\">";
  msg += "Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn off led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn on led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"BLINK\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to blink led";
  msg += "</form>";
  
  msg += "<br><hr>IP address: ";
  msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";

  msg += "</body>";
  msg += "</html>";
  // end of webpage file

  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}

void setup() {
  Serial.begin(115200);
  Serial.println();
  pinMode(statusLed, OUTPUT);
  ledStatus = LEDBLINK;
  
  // optional to set ip address of AP
  IPAddress APip(1, 2, 3, 4);
  IPAddress APgateway(1, 2, 3, 4);  // the ip address and the gateway have to be the same
  IPAddress APsubnet(255, 255, 255, 0);
  WiFi.softAPConfig(APip, APgateway, APsubnet);

  WiFi.softAP(ssid);              // start AP without password
  // WiFi.softAP(ssid, password); // start AP with password
  
  IPAddress IP = WiFi.softAPIP();
  WiFi.mode(WIFI_AP);      // use just AP mode

  Serial.print("AP ssid: ");
  Serial.println(ssid);
//  Serial.print("AP password: ");
//  Serial.println(password);
  Serial.print("AP IP address: ");
  Serial.println(IP);

  server.on("/", indexHTML);  // set root webpage
  server.begin(); // Start server
}

void loop() {
  flashActivityLed();
  server.handleClient();  // handle web client request
  yield();  // must have this to give time to handle wifi background stuff 
}
