/* //////////////////////////////////////////////////////////
 * A temperature and humidity Web Server using the ESP8266
 * and the DHT11 temperature sensor.
 *
 * Connect the Out pin on the DHT11 temperature sensor to pin D2 on the ESP8266.
 * Connect the + and - pins on the DHT11 to +5V and GND, respectively.
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "Temperature AP"
 * then browse to the URL
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

 
//////////////////////////////////////////////////////////////
//// status LED stuff
#define statusLed 1     // for ESP8266-01
//#define statusLed 2     // for ESP8266-12
bool statusLedOn = false;

// Flashes the led without using the delay() function.
// The period of each cycle is flashDelay milliseconds
// flashSpeed determines the speed of each flash
// flashTimes determines the number of times to flash in a cycle
// This version needs to be called inside a loop in order to flash
// In order to notice a pause between each cycle
// flashDelay should be greater than 2*flashSpeed*flashTimes
//  e.g.   flashActivityLed(1000, 100, 1);
void flashActivityLed(int flashDelay=3000, int flashSpeed=150, int flashTimes=2) {
  static unsigned long nextLedActivity = 0;
  static int t = 0;
  if (statusLedOn) return;
  if ((millis() > nextLedActivity)) {
    if (t == 0) {
      digitalWrite(statusLed, LOW); // turn on LED
    } else {
      digitalWrite(statusLed, !digitalRead(statusLed)); // toggle LED
    }
    if (t < flashTimes*2-1) {
      nextLedActivity = millis() + flashSpeed;
      t++;
    } else {
      nextLedActivity = millis() + flashDelay - (flashSpeed*2*flashTimes);
      t = 0;
    }
  }
}


//////////////////////////////////////////////////////////////
//// DHT11 temperature stuff
#include <DHT.h>
#define DHTPIN 2  // data in from DHT11
#define DHTTYPE DHT11
DHT dht(DHTPIN, DHTTYPE);

//////////////////////////////////////////////////////////////
//// Webserver stuff
void handleSTAClient() {
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<meta http-equiv=\"refresh\" content=\"2\">"; // auto refresh page every 2 seconds
  
  msg += "<title>ESP8266 Demo</title>";
  msg += "  <style>";
  msg += "    body { background-color: #cccccc; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "  <h1>Hello from RobotsForFun!</h1>";
  msg += "  This is an ESP8266 demo webserver that tells you the temperature and humidity.";

  // read the temperature and humidity
  long c, f, h;
  while(1) {
    c = dht.readTemperature();  // Celsius
    f = dht.readTemperature(true);  // Fahrenheit
    h = dht.readHumidity();
    if (!isnan(c) && !isnan(f) && !isnan(h)) break;
    delay(250);
  }
  if (c < 200) {
    Serial.print(c);
    Serial.print(", ");
    Serial.print(f);
    Serial.print(", ");
    Serial.println(h);
    
    msg += "<br><br>The temperature is ";
    msg += String(c);
    msg += " degrees Celsius, which is ";
    msg += String(f);
    msg += " degrees Fahrenheit";
    msg += "<br>The humidity is ";
    msg += String(h);
    msg += "%";
  } else {
    Serial.println("The DHT temperature module is malfunctioning or not connected.");
    msg += "<br><br>The DHT temperature module is malfunctioning or not connected.";
  }

  msg += "<br><br><hr>";  
  msg += "Go to <a href=\"http://www.RobotsForFun.com\">RobotsForFun.com</a> for more interesting projects.";

  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send ( 200, "text/html", msg);
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);

  setupAP("Temperature AP");  // setup access point using ssid "Temperature AP" and no password
  char devicename[] = "ESP8266 Demo Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  setupWiFi(devicename);  // use DHCP

  //// DHT Temperature stuff
  dht.begin();
}

void loop() {
  flashActivityLed();

  if (APisOn && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  server.handleClient();
  yield();  // yield time for the webserver to handle the clients
}
