/* //////////////////////////////////////////////////////////
 * A Basic Switch Web Server using the ESP8266
 * that tells you whether a switch is turned on or off
 *
 * Connect a switch input to pin 12 for the ESP8266-12
 *                        or pin  2 for the ESP8266-01
 * The default is active low for this input.
 * You can change it to active high in the code below
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or to the one you provided, then browse to the IP address
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to go to
 * 1.2.3.4/setup to setup the ssid and password
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */


//////////////////////////////////////////////////////////////
//// status LED stuff
//#define ESP8266_01  // define this when using the ESP8266-01
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01
#else
  #define statusLed 2     // for ESP8266-12
#endif


//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include "DateTime_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// WiFi stuff
#include "WiFi_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// switch stuff
#ifdef ESP8266_01
  #define switchPin 2     // for ESP8266-01
#else
  #define switchPin 12    // for ESP8266-12
#endif
#define ACTIVE_HIGH 1 // for active high switch
#define ACTIVE_LOW 0  // for active low switch
bool ActiveMode = ACTIVE_LOW; // set active high or low switch mode
bool switchOn = false;
int count = 0;

String turnOnMsg="";
String turnOffMsg="";

// for active HIGH input
void checkSwitch() {
  if (digitalRead(switchPin) == ActiveMode) {
    // switch turned on
    if (!switchOn) {
      switchOn = true;
      digitalWrite(statusLed, LOW); // turn on LED
      ledStatus = true;
      turnOnMsg = "Switch turned on at " + formattedTime() + " count=" + String(++count);
      Serial.println(turnOnMsg);
    }
  } else {
    // switch turned off
    if (switchOn) {
      switchOn = false;
      digitalWrite(statusLed, HIGH);  // turn off LED
      ledStatus = false;
      turnOffMsg = "Switch turned off at " + formattedTime();
      Serial.println(turnOffMsg);
    }
  }
  
}


//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<meta http-equiv=\"refresh\" content=\"1\">"; // auto refresh page every 1 second
  
  msg += "<title>ESP8266 Demo</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Switch Web Server</b></font><br>";
  msg += "  This is an ESP8266 demo webserver that tells you whether a switch is turned on or off.";
  msg += "<br>";
  msg += formattedTime();
  msg += "</center>";
  msg += "<hr><br>";

  msg += "The switch is currently ";
  if (switchOn) {
    msg += "ON";
  } else {
    msg += "OFF";
  }
  msg += "<br><br>";
  msg += turnOnMsg;
  msg += "<br>";
  msg += turnOffMsg;

  msg += "<br><br><a href=\"/?RESETCOUNT=ON\"><button style=\"font-size:16pt\">Reset Count</button></a>";
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";

  // send webpage to client
  server.send(200, "text/html", msg);

  if (server.hasArg("RESETCOUNT")) {
    Serial.println("Reseting count to 0");
    count = 0;
  }  
}

void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  //pinMode(switchPin, INPUT);           // for input with no pull-up or pull-down
  if (ActiveMode == ACTIVE_HIGH) {
    pinMode(switchPin, OUTPUT);          // for active HIGH input with internal pull-down
  } else {
    pinMode(switchPin, INPUT_PULLUP);    // for active LOW input with internal pull-up
  }

  setupAP();  // setup access point using default ssid "WiFi setup ESP8266-xxxx" and no password
  char devicename[] = "ESP8266 Demo Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  setupWiFi(devicename);  // use DHCP

  setClock(PST);

  // update systemStartTime and systemStartCount
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
}

void loop() {
  checkSwitch();
  
  if (APisOn && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();
  server.handleClient();
  // yield();  // don't need this because yield() is called automatically after the loop() function exits
}
