/*
 * ESP8266 SPIFFS Sample Webserver for the ESP8266-12
 * 
 * This ESP8266 webserver serves HTML and image files that are stored in SPIFFS.
 * The files are first uploaded to the ESP flash memory using SPIFFS tool.
 * 
 * Need to first upload all the files in the data folder to flash memory
 * using the SPIFFS ESP8266 Sketch Upload Tool
 * 
 * Once the webserver is up, browse to the IP address shown in the serial monitor
 * Examples:
 *   192.168.1.114
 *   192.168.1.114/index.html
 *   192.168.1.114/image.png
 *   192.168.1.114/readme.txt
 *   192.168.1.114/image.png?download // to download the file
 *   192.168.1.114/asdf.jkl
 * 
 * Reference:
 *   https://circuits4you.com/2018/01/31/upload-image-png-jpeg-to-esp8266-web-page/
 *   
 * Copyright 2020 Enoch Hwang
 */


//////////////////////////////////////////////////////////////
//// SPIFFS stuff
#include <FS.h>   //Include File System Headers

void listAllFiles() {
  Dir dir = SPIFFS.openDir("");
//  Dir dir = SPIFFS.openDir("/");
  while (dir.next()) {
    Serial.print(dir.fileName());
    Serial.print(" ");
    Serial.println(dir.fileSize());
    }
}

// initialize the SPIFFS file system
bool initSPIFFS() {
  if (SPIFFS.begin()) {
    Serial.println("SPIFFS initialized successfully");
    listAllFiles();
    return true;  
  } else {
    Serial.println("Error initializing SPIFFS");
    return false;
  }
}


//////////////////////////////////////////////////////////////
//// WiFi stuff
#define NOTFOUNDHTML  // define this to use your own notfoundHTML() function
#include "WiFi_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// Webserver HTML webpage stuff

bool loadFromSPIFFS(String path){
  String dataType = "text/plain";
  if (path.endsWith("/")) path += "index.html";

  if (path.endsWith(".src")) path = path.substring(0, path.lastIndexOf("."));
  else if (path.endsWith(".html")) dataType = "text/html";
  else if (path.endsWith(".htm")) dataType = "text/html";
  else if (path.endsWith(".css")) dataType = "text/css";
  else if (path.endsWith(".js")) dataType = "application/javascript";
  else if (path.endsWith(".png")) dataType = "image/png";
  else if (path.endsWith(".gif")) dataType = "image/gif";
  else if (path.endsWith(".jpg")) dataType = "image/jpeg";
  else if (path.endsWith(".ico")) dataType = "image/x-icon";
  else if (path.endsWith(".xml")) dataType = "text/xml";
  else if (path.endsWith(".pdf")) dataType = "application/pdf";
  else if (path.endsWith(".zip")) dataType = "application/zip";
  if (server.hasArg("download")) dataType = "application/octet-stream"; // if URL has has parameter "?download" this will download the file instead

  // open the file in SPIFFS for reading
  File dataFile = SPIFFS.open(path.c_str(), "r");
  if (!dataFile) {
    // open failed
    dataFile.close();
    return false;
  }

  // send file to client
  if (server.streamFile(dataFile, dataType) != dataFile.size()) {
  }

  dataFile.close();
  return true;
}


void indexHTML() {
  loadFromSPIFFS("/index.html");  // load the index.html file from SPIFFS and send it to the client
}

// another way of doing it
//void indexHTML() {
//  server.sendHeader("Location", "/index.html", true);   //Redirect to our html web page
//  server.send(302, "text/plane", "");
//}

// for all files, first try to load it from SPIFFS
// if file exists then send it back to the client
// otherwise send a 404 file not found error
void notfoundHTML() {
  if (loadFromSPIFFS(server.uri())) {
    return;
  }
  String message = "File Not Found Error\n";
  message += " URI: ";    // Uniform Resource Identifier
  message += server.uri();
  message += "\n Method: ";
  message += (server.method() == HTTP_GET)?"GET":"POST";
  message += "\n Arguments: ";
  message += server.args();
  message += "\n";
  for (uint8_t i=0; i<server.args(); i++){
    message += " NAME:"+server.argName(i) + "\n VALUE:" + server.arg(i) + "\n";
  }
  server.send(404, "text/plain", message);
  Serial.println(message);
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  Serial.println();
  
  // SPIFFS stuff
  initSPIFFS();

  // Wifi stuff
  setupWiFi();
}

void loop() {
  flashActivityLed();     // show system is alive
  server.handleClient();
}
