/* //////////////////////////////////////////////////////////
 * A Web Server using the ESP8266
 * that allows you to control a WS2812 RGB led light string/ring.
 *
 * Connect the DI pin on the WS2812 RGB LED light string/ring
 * to pin 13 on the ESP8266-12
 * or to pin 2 on the ESP8266-01
 * 
 * Change the led count in #define LED_COUNT 150 to match your led string/ring
 * Uncomment #define ESP8266_01 if using the ESP8266-01
 * 
 * After uploading this program to the ESP8266 module,
 * open the monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or to the one you provided, then browse to the IP address
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to go to
 * 1.2.3.4/setup to setup the ssid and password
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright 2016 Enoch Hwang
 */


//////////////////////////////////////////////////////////////
//// VCC voltage stuff
  ADC_MODE(ADC_VCC);


//////////////////////////////////////////////////////////////
//// status LED stuff
//#define ESP8266_01  // define this when using the ESP8266-01
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01
#else
  #define statusLed 2     // for ESP8266-12
#endif


//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include "DateTime_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// WiFi stuff
#include "WiFi_RobotsForFun.h"



//////////////////////////////////////////////////////////////
//// WS2812 RGB LED stuff
#ifdef ESP8266_01
  #define DI_PIN 2
#else
  #define DI_PIN 13
#endif
#define LED_COUNT 150
#include "WS2812.h"
WS2812 leds = WS2812(LED_COUNT, DI_PIN, NEO_GRB + NEO_KHZ800);
#include "WS2812patterns.h"


//////////////////////////////////////////////////////////////
// Do delay without using the delay() function
// wait <0 will block until exitNow is true
//      == will return immediately
//      >0 returns after wait milliseconds
void nonblockingDelay_WS2812(int wait) {
  if (wait == 0) return;
  else if (wait < 0) {
    while (!exitNow) {
      flashActivityLed();
      server.handleClient();
      yield();
    }
  } else {
    unsigned long startTime = millis();  
    while (millis() - startTime < wait && !exitNow) {
      flashActivityLed();
      server.handleClient();
      yield();
    }
  }
}

// RGB lights modes
#define OFF 0
#define SOLID 1
#define WIPE 2
#define RAINBOW 3
#define BLOCKS 4
#define CHASE 5
#define CHASERAINBOW 6
#define SPARKLES 7
#define DISCO 8
#define MOVEIN 9
#define FLASH 10
#define FADE 11
#define FADERAINBOW 12
#define CYCLONE 13
#define SINGLE 14
#define DEMO 15
#define NONE 16
unsigned int RGBLightsMode = OFF;
byte red, green, blue;
int numberOfLeds, wait, times;

void checkLights() {
  exitNow = false; // to allow early termination of the RGB loops
  switch(RGBLightsMode) {
    case SOLID:
      allOn(leds, nonblockingDelay_WS2812, Color(red,green,blue), -1);  // the -1 is for blocking
      break;
    case RAINBOW:
      rainbow(leds, nonblockingDelay_WS2812, 0);
      break;
    case WIPE:
      wipe(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case BLOCKS:
      blocks(leds, nonblockingDelay_WS2812);
      break;
    case CHASE:
      theaterChase(leds, nonblockingDelay_WS2812, Color(red,green,blue), 0);
      break;
    case CHASERAINBOW:
      RGBLightsMode = OFF; // turn off lights
      break;
    case SPARKLES:
      sparkles(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case DISCO:
      discoLights(leds, nonblockingDelay_WS2812);
      break;
    case MOVEIN:
      moveIn(leds, nonblockingDelay_WS2812);
      break;
    case FLASH:
      flash(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case FADE:
      fade(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case FADERAINBOW:
      fadingRainbow(leds, nonblockingDelay_WS2812, 0);
      break;
    case CYCLONE:
      cyclone(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case SINGLE:
      single(leds, nonblockingDelay_WS2812, Color(red,green,blue));
      break;
    case DEMO:
      wipe(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      rainbow(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      flash(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      blocks(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      fade(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      theaterChase(leds, nonblockingDelay_WS2812); 
      if (exitNow) break;
      cyclone(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      sparkles(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      moveIn(leds, nonblockingDelay_WS2812);
      if (exitNow) break;
      discoLights(leds, nonblockingDelay_WS2812);
      RGBLightsMode = DEMO;
      break;
    case OFF:
      allOff(leds);
      RGBLightsMode = NONE;
      break;
    default:
      break;
  }
}


//////////////////////////////////////////////////////////////
//// Clock chime stuff
void checkClock() {
  int s = second();
  int m = minute();
  int h = hour();
  if ((m == 0) && (s == 0)) {
    for (int i=0; i<hour12(); i++) {
      allOn(leds, nonblockingDelay_WS2812, Color(0,0,255), 0);
      nonblockingDelay_WS2812(500);
      allOff(leds);
      nonblockingDelay_WS2812(500);
    }
    rainbow(leds, nonblockingDelay_WS2812, 10000);
    allOff(leds);
  } else if ((m == 15) && (s == 0)) {
    rainbow(leds, nonblockingDelay_WS2812, 5000);
    allOff(leds);
  } else if ((m == 30) && (s == 0)) {
    rainbow(leds, nonblockingDelay_WS2812, 5000);
    allOff(leds);
  } else if ((m == 45) && (s == 0)) {
    rainbow(leds, nonblockingDelay_WS2812, 5000);
    allOff(leds);
  }
}


//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  String msg = "";

  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
//  msg += "<meta http-equiv=\"refresh\" content=\"1\">"; // auto refresh page every 1 second
  
  msg += "<title>Rainbow Lights</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Rainbow Lights</b></font><br>";
  msg += "This is an ESP8266 demo webserver that allows you to control a RGB led ring/string.";
  msg += "<br>" + formattedTime();
  msg += "</center>";
  msg += "<hr>";

  // create the buttons
  msg += "<form style=\"font-size:14pt\">";
  msg += "Select color:";
  msg += "<br>&nbsp;&nbsp;<input type=\"color\" name=\"COLOR\" id=\"myrgb\" value=\"#xxxxxx\" />";
  msg += "<br><br>";
  msg += "Select pattern:";
  msg += "<table>";
  msg += "<tr><td>&nbsp;&nbsp;<input type=\"submit\" name=\"pattern\" value=\"SOLID\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"CHASE\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"WIPE\" style=\"font-size:14pt\" >";
  msg += "<tr><td>&nbsp;&nbsp;<input type=\"submit\" name=\"pattern\" value=\"BLOCKS\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"FLASH\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"DISCO\" style=\"font-size:14pt\" >";
  msg += "<tr><td>&nbsp;&nbsp;<input type=\"submit\" name=\"pattern\" value=\"RAINBOW\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"FADE\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"MOVEIN\" style=\"font-size:14pt\" >";
  msg += "<tr><td>&nbsp;&nbsp;<input type=\"submit\" name=\"pattern\" value=\"SPARKLES\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"CYCLONE\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"SINGLE\" style=\"font-size:14pt\" >";
  msg += "<tr><td>&nbsp;&nbsp;<input type=\"submit\" name=\"pattern\" value=\"DEMO\" style=\"font-size:14pt\" >";
  msg += "<td><input type=\"submit\" name=\"pattern\" value=\"OFF\" style=\"font-size:14pt\" >";
  msg += "</table>";
//  msg += "<br>";
//  msg += "Select options:";
//  msg += "<table>";
//  msg += "<tr><td>&nbsp;&nbsp;Color:<td><input type=\"color\" name=\"COLOR\" id=\"myrgb\" value=\"#xxxxxx\" />";
////  msg += "<tr><td>&nbsp;&nbsp;Number of leds:<td><input type=\"number\" name=\"number\" min=\"1\" max=\"150\" style=\"font-size:14pt\" >";
//  msg += "<tr><td>&nbsp;&nbsp;Speed:<td><input type=\"number\" name=\"wait\" min=\"1\" max=\"500\" style=\"font-size:14pt\" >";
//  msg += "<tr><td>&nbsp;&nbsp;Number of times:<td><input type=\"number\" name=\"times\" min=\"1\" max=\"150\" style=\"font-size:14pt\" >";
//  msg += "</table>";
  msg += "</form>";

  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "<br><br><hr>";  
  msg += "Go to <a href=\"http://www.RobotsForFun.com\">RobotsForFun.com</a> for more interesting projects.";

  msg += "</body>";
  msg += "</html>";

  // get the color input to update the webpage
  String color;
  if (server.argName(0).equals("COLOR")) {  // make sure the argument name is "COLOR"
    color = server.arg(0);
    msg.replace("#xxxxxx",color);
  }
  
  // send webpage to browser
  server.send ( 200, "text/html", msg);
  
  // get the selected color. convert color string in hex to decimal
  char s[4];
  color.substring(1,3).toCharArray(s,sizeof(s));  // convert String to c-string
  red = strtol(s,NULL,16);                        // convert hex string to value
  color.substring(3,5).toCharArray(s,sizeof(s));
  green = strtol(s,NULL,16);
  color.substring(5,7).toCharArray(s,sizeof(s));
  blue = strtol(s,NULL,16);
  
  // get url arguments and process the command that DO have time delays
  if (server.argName(2).equals("number")) {
    numberOfLeds = server.arg(2).toInt();
  }
  if (server.argName(2).equals("wait")) {
    wait = server.arg(2).toInt();
  }
  if (server.argName(3).equals("times")) {
    times = server.arg(3).toInt();
  }
  
  if (server.arg("pattern").equals("SOLID")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("solid ");
    RGBLightsMode = SOLID;
  } else if (server.arg("pattern").equals("WIPE")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("wipe ");
    RGBLightsMode = WIPE;
  } else if (server.arg("pattern").equals("RAINBOW")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("rainbow ");
    RGBLightsMode = RAINBOW;
  } else if (server.arg("pattern").equals("BLOCKS")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("blocks ");
    RGBLightsMode = BLOCKS;
  } else if (server.arg("pattern").equals("CHASE")) {
    exitNow = true;  // to allow early termination of the RGB loops
//    if (red == 0 && green == 0 && blue == 0) {
//      Serial.print("chaserainbow ");
//      RGBLightsMode = CHASERAINBOW;
//    } else {
//      Serial.print("chase ");
//      RGBLightsMode = CHASE;
//    }
    RGBLightsMode = CHASE;
  } else if (server.arg("pattern").equals("SPARKLES")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("sparkles ");
    RGBLightsMode = SPARKLES;
  } else if (server.arg("pattern").equals("DISCO")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("disco ");
    RGBLightsMode = DISCO;
  } else if (server.arg("pattern").equals("MOVEIN")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("movein ");
    RGBLightsMode = MOVEIN;
  } else if (server.arg("pattern").equals("FLASH")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("flash ");
    RGBLightsMode = FLASH;
  } else if (server.arg("pattern").equals("FADE")) {
    exitNow = true;  // to allow early termination of the RGB loops
    if (red == 0 && green == 0 && blue == 0) {
      Serial.print("faderainbow ");
      RGBLightsMode = FADERAINBOW;
    } else {
      Serial.print("fade ");
      RGBLightsMode = FADE;
    }
  } else if (server.arg("pattern").equals("CYCLONE")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("cyclone ");
    RGBLightsMode = CYCLONE;
  } else if (server.arg("pattern").equals("SINGLE")) {
    exitNow = true;  // to allow early termination of the RGB loops
    if (times == 0) times = 1;
    Serial.print("single ");
    RGBLightsMode = SINGLE;
  } else if (server.arg("pattern").equals("DEMO")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("demo ");
    RGBLightsMode = DEMO;
  } else if (server.arg("pattern").equals("OFF")) {
    exitNow = true;  // to allow early termination of the RGB loops
    Serial.print("off ");
    RGBLightsMode = OFF;
  }  
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  delay(200);
  
  //// RGB LED stuff
  leds.begin();      // Call this to start up the LED strip.
  leds.clear();      // turns all LEDs off
  leds.setBrightness(100);        // 0=off; 255=full bright
  leds.setPixelColor(0, 0,0,100); // turn on blue on pixel 0
  leds.setPixelColor(1, 0,0,100); // turn on blue on pixel 1
  leds.show();

  //// set clock time zone to PST with automatic hourly updates and daylight savings time adjustments
  setClock(PST);

  //// WiFi stuff
  char ssid[] = "your ssid";           // change this to your ssid
  char password[] = "your password";   // change this to your password
  char devicename[] = "Rainbow Lights"; // optional. change to whatever you want. Use "ping ESP8266" to test
  IPAddress staticIP(192, 168, 1, 99); // use this if you want a static IP address
  setupAP(devicename);  // setup access point using given device name

  bool connected;
  
  connected = setupWiFi(devicename, ssid, password);
  
  // first try to connect to the ssid and password obtained from EEPROM
  // if unsuccessful then try to connect to the ssids and passwords passed in
  // const int NUMBER_OF_SSIDS = 4;
  // const char default_ssids[NUMBER_OF_SSIDS][SSID_SIZE] = {"ssid_1", "ssid_2", "ssid_3", "ssid_4"};
  // const char default_passwords[NUMBER_OF_SSIDS][PASSWORD_SIZE] = {"password_1", "password_2", "password_3", "password_4"};
  // connected = setupWiFi(devicename, default_ssids, default_passwords, NUMBER_OF_SSIDS);   // use DHCP and no blocking

  if (connected) {
    leds.setPixelColor(0, 0,100,0); // turn on green on pixel 0
    leds.setPixelColor(1, 0,100,0); // turn on green on pixel 1
  } else {
    leds.setPixelColor(0, 100,0,0); // turn on red on pixel 0
    leds.setPixelColor(1, 100,0,0); // turn on red on pixel 1
  }
  leds.show();

  unsigned long startTime = millis(); // for keeping the IP address on for 5 seconds if connected

  // wait for clock to sync with server
  #ifdef __DateTime_RobotsForFun__
    unsigned long displayTime = millis();
    if (internetIsConnected) {
      Serial.println();
      if (waitForTimeValid()) {
        Serial.print("Current time is ");
        Serial.println(formattedTime());
      } else {
        Serial.println("Failed to connect to NTP server");
      }
    }
  #endif

  while(millis() - startTime < 3000) {  // for keeping the IP address on for 5 seconds if connected
    flashActivityLed(3000,150,1);
    server.handleClient();
    yield();  // must have this to give time to handle wifi background stuff
  }

  // update systemStartTime and systemStartCount
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  

  leds.clear();      // turns all LEDs off
}


void loop() {
  checkLights();
  checkClock();  
  
  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();     // show system is alive
  server.handleClient();  // handle web client request
  //yield();  // yield() is called automatically at the end of loop() must have this to give time to handle wifi background stuff
}
