/*
 * An access point (AP) webserver using the ESP8266
 * that allows you to turn on and off a LED.
 *
 * Connect negative leg of LED to D2.
 * Connect positive leg of LED to 150 ohm resistor and then to VCC.
 *
 * Connect your WiFi to the given name (default is Led Webserver)
 * and then browse to 1.2.3.4
 * 
 * Copyright 2016 Enoch Hwang
 */
/* //////////////////////////////////////////////////////////
 * An access point (AP) only Web Server using the ESP8266-12
 * to turn on and off a LED
 *
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "ESP8266 Demo Webserver"
 * then browse to the URL
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

//#define ESP8266_01  // define this when using the ESP8266-01
                      // otherwise it defaults to using the ESP8266-12

//////////////////////////////////////////////////////////////
//// VCC voltage stuff
ADC_MODE(ADC_VCC);

//////////////////////////////////////////////////////////////
//// status LED stuff
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01. Using this inteferes with the Serial monitor, i.e. nothing will be printed
#else
  #define statusLed 2     // for ESP8266-12 and ESP32
#endif

//////////////////////////////////////////////////////////////
//// WiFi stuff
#include <WiFi_RobotsForFun.h>

//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("LED")) {
    if (String(server.arg("LED")) == "OFF") {
      digitalWrite(statusLed, LEDOFF);  // turn off LED
      ledStatus = LEDOFF;
      Serial.println("turn off LED");
    } else if (String(server.arg("LED")) == "ON") {
      digitalWrite(statusLed, LEDON);   // turn on LED
      ledStatus = LEDON;
      Serial.println("turn on LED");
    } else if (String(server.arg("LED")) == "BLINK") {
      ledStatus = LEDBLINK;                // blink LED
      Serial.println("blink LED");
    }
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<title>ESP8266 Demo</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>Led Webserver</b></font><br>";
  msg += "This is an ESP8266 demo webserver that allows you to turn on and off a LED.";
  msg += "</center>";
  msg += "<hr><br>";

  msg += "Current led status is ";
  switch(ledStatus) {
    case LEDOFF: 
      msg += "<b>OFF</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=ON\"><button style=\"font-size:16pt\">ON</button></a>";
      break;
    case LEDON: 
      msg += "<b>ON</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=BLINK\"><button style=\"font-size:16pt\">BLINK</button></a>";
      break;
    default: 
      msg += "<b>BLINK</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a>";
      break;
  }

  // another way to create the buttons
  msg += "<br><br>Another way to do it:";
  msg += "<form style=\"font-size:12pt\">";
  msg += "Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn off led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn on led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"BLINK\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to blink led";
  msg += "</form>";
  
  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send( 200, "text/html", msg);

  // get url arguments and process the command
  // changes here will not be reflected in the current webpage
  // can have long time delays since the webpage has already been sent back to the client

}

////////////////////////////////////////////////////////////////
////// HTML webpage 
//void indexHTML2() {
//  String msg = "";
//  msg += "<!DOCTYPE html>";
//  msg += "<head>";
//  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
//  
//  msg += "<title>ESP8266 Demo</title>";
//  msg += "  <style>";
//  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
//  msg += "  </style>";
//  msg += "</head>";
//  
//  msg += "<body>";
//  msg += "<h1>Hello from RobotsForFun!</h1>";
//  msg += "This is an ESP8266 demo webserver that allows you to turn on and off a LED.";
//  msg += "<br><br>";
//
//  // create two buttons
//
//  msg += "  <form style=\"font-size:16pt\">";
//  msg += "  <input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
//  msg += "  <input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
//  msg += "  </form>";
//
//  // another way to create two buttons
//  msg += "<br><br>Another way to create two buttons";
//  msg += "<br><br>&nbsp;&nbsp;Click <a href=\"/\?LED=ON\"><button style=\"font-size:16pt\">ON</button></a> to turn it on<br>";
//  msg += "<br>&nbsp;&nbsp;Click <a href=\"/\?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a> to turn it off<br>";
//
//  msg += "<br><hr>Signal strength: ";
//  msg += signalStrength();
//  msg += "%";
//  msg += "<br>Vcc voltage: ";
//  int p = (ESP.getVcc() + 5) / 10;
//  msg += String(p/100.0);
//  msg += "V";
//  msg += "<br>IP address: ";
//  msg += WiFi.localIP().toString();
//  msg += "<br>Hostname: ";
//  msg += WiFi.hostname();
//
//  msg += "<br>Setup Wifi: ";
//  String m = "";
//  m += WiFi.localIP()[0];
//  m += ".";
//  m += WiFi.localIP()[1];
//  m += ".";
//  m += WiFi.localIP()[2];
//  m += ".";
//  m += WiFi.localIP()[3];
//  msg += "<a href=\"http://" + m + "/setup\">" + m + "/setup</a>";
//
//  msg += "</body>";
//  msg += "</html>";
//
//  // send webpage to browser
//  server.send( 200, "text/html", msg);
//
//  
//  // get url arguments and process command
//  if (server.hasArg("LED")) {
//    if (String(server.arg("LED")) == "ON") {
//      digitalWrite(statusLed, LOW);  // turn on LED
//      statusLedOn = true;
//      Serial.println("turn on LED");
//    } else if (String(server.arg("LED")) == "OFF") {
//      digitalWrite(statusLed, HIGH);  // turn on LED
//      statusLedOn = false;
//      Serial.println("turn off LED");
//    }
//  }
//}



void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);

  char APssid[SSID_SIZE] = "Led Webserver";
  char APpassword[PASSWORD_SIZE] = "";  // empty string for no password
  // setupAP(APssid, APpassword); // setup access point with ssid and password
  setupAP(devicename);            // setup access point using devicename
  //setupAP();                    // setup access point using default device name  
}

void loop() {

  flashActivityLed();     // show system is alive
  server.handleClient();
  // yield();  // don't need this because yield() is called automatically after the loop() function exits
}
