/*
 * Digital clock using either 4-digit x 7-segment or 8-digit 16-segment LED displays
 * 
 * MAX7219 connections (for 8 digits)
 * DIN  to pin D14
 * CLK  to pin D13
 * CS   to pin D12
 * 
 * MAX7219 connections (for 4 digits)
 * DIN  to pin D14
 * CLK  to pin D12
 * CS   to pin D13
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address.
 * 
 * or
 * 
 * Connect your WiFi to the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or to the one you provided, then browse to the IP address
 *     1.2.3.4 
 * for the main page or
 *     1.2.3.4/setup
 * for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to go to
 * 1.2.3.4/setup to setup the ssid and password
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the status led on pin 1
 * 
 * Copyright 2017, 2020 Enoch Hwang
 */

 
//////////////////////////////////////////////////////////////
//// VCC voltage stuff
  ADC_MODE(ADC_VCC);


//////////////////////////////////////////////////////////////
//// status LED stuff
#define statusLed 2     // for ESP8266-12


//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include "DateTime_RobotsForFun.h"


//////////////////////////////////////////////////////////////
//// WiFi stuff
#include "WiFi_RobotsForFun.h"

// Do delay without using the delay() function
void nonblockingDelay(int wait) {
  unsigned long timesUp;
  timesUp = millis() + wait;
  while (millis() < timesUp) {
//    flashActivityLed();
    server.handleClient();
    yield();  // must have this to give time to handle wifi background stuff
  }
}


//////////////////////////////////////////////////////////////
//// HTML webpage 
void indexHTML() {
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  msg += "<meta http-equiv=\"refresh\" content=\"1\">"; // auto refresh page every 1 second
  
  msg += "<title>LED Clock</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<center><font size = \"3\"><b>LED Network Clock</b></font><br>";
  msg += "Real time clock for the<br>8x16 or 4x7 segment LED display<br>using the MAX7219 controller";
  msg += "<br><b>" + formattedTime() + "</b>";
  msg += "</center>";
  msg += "<hr>";

  if (!isTimeValid()) {
    msg += "<form style=\"font-size:12pt\">";
    msg += "&nbsp;Year: <input type=\"number\" name=\"YR\" min=\"1990\" max=\"2035\" maxlength=\"4\" style=\"width:60px;font-size:12pt\" >";
    msg += "&nbsp;Month: <input type=\"number\" name=\"MO\" min=\"1\" max=\"12\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
    msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Day: <input type=\"number\" name=\"DA\" min=\"1\" max=\"31\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
    msg += "<br>Hour: <input type=\"number\" name=\"HR\" min=\"0\" max=\"23\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
    msg += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Minute: <input type=\"number\" name=\"MI\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
    msg += "&nbsp;Second: <input type=\"number\" name=\"SE\" min=\"0\" max=\"59\" maxlength=\"2\" style=\"width:40px;font-size:12pt\" >";
    msg += "<br><input type=\"submit\" value=\"Set clock\" style=\"font-size:16pt\">";
    msg += "</form>";
    msg += "<hr>";
  }
  
  // wifi setup and info links
  msg += "IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }

  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command that DO have time delays
  if (server.hasArg("YR")) {
    // set the clock
    int yr, mo, da, hr, mi, se;
    yr = server.arg("YR").toInt();
    mo = server.arg("MO").toInt();
    da = server.arg("DA").toInt();
    hr = server.arg("HR").toInt();
    mi = server.arg("MI").toInt();
    se = server.arg("SE").toInt();
    if (yr == 0) yr = year();
    if (mo == 0) mo = month();
    if (da == 0) da = day();
    if (hr == 0) hr = hour();
    if (mi == 0) mi = minute();
    setClock(yr, mo, da, hr, mi, se);
  }
}


//////////////////////////////////////////////////////////////
//// MAX7219 16x8 and 7x4 LED display stuff
//#define DISPLAY_8_DIGIT    // 8 digit x 16-segment digit display
#define DISPLAY_4_DIGIT    // 4 digit x 7-segment digit display

#include "LedControl.h"
// MAX7219 connections for the 16x8 display
#ifdef DISPLAY_8_DIGIT
  #define DIN 14
  #define CLK 13
  #define CS  12
#endif
// MAX7219 connections for the 7x4 display
#ifdef DISPLAY_4_DIGIT
  #define DIN 14
  #define CLK 12
  #define CS  13
#endif
#define NO 1  // number of MAX72XX devices connected
LedControl led=LedControl(DIN,CLK,CS,NO);
int brightness = 15; // max bright
bool hr12 = true;

void displayClock() {
  const static char w10[7] = {'S','M','T','W','T','F','S'};  // The "T" is changed in the LedControl.h definition
  const static char w1[7]  = {'u','o','u','d','h','r','A'};

  time_t t = time(nullptr);   // get current date/time
  struct tm st = *gmtime(&t); // convert to time structure
  int h = hour12(st);         // extract the parts
  int h24 = hour(st);
  int m = minute(st);
  int s = second(st);
  int w = weekday(st);        // 0=Sunday
  int mo = month(st);
  int d = day(st);
  
  int h10 = h/10;             // separate the 10th and unit digits
  int h1 = h - h10*10;
  int m10 = m/10;
  int m1 = m - m10*10;
  int s10 = s/10;
  int s1 = s - s10*10;
  int mo10 = mo/10;
  int mo1 = mo - mo10*10;
  int d10 = d/10;
  int d1 = d - d10*10;

  // set brightness depending on time of day
  if (h24 < 5) led.setIntensity(0,1);  // set the brightness
  else if (h24 >= 5 && h24 <= 22) led.setIntensity(0,15);
  else led.setIntensity(0,7);

  #ifdef DISPLAY_8_DIGIT
  if (s1 < 7) { // display hour,minute,second
    led.setDigit(0,7,h10,false);
    led.setDigit(0,6,h1,false);
    led.setChar (0,5,' ',true);
    led.setDigit(0,4,m10,false);
    led.setDigit(0,3,m1,false);
    led.setChar (0,2,' ',true);
    led.setDigit(0,1,s10,false);
//    if (hr12 && isPM())
//      led.setDigit(0,0,s1,false);
//    else
//      led.setDigit(0,0,s1,false);
    led.setDigit(0,0,s1,false);
    nonblockingDelay(500); // flash the dot
    led.setChar(0,5,' ',false);
    led.setChar(0,2,' ',false);    
  } else { // display weekday,month,day
    if (w >= 1 && w <= 4) { // the character 'T' in the LEDControl library is changed for this
      led.setChar(0,7,w10[w],true);  // turn on DP (middle bar)
    } else {
      led.setChar(0,7,w10[w],false);
    }
    led.setChar(0,6,w1[w],false);
    led.setChar (0,5,'-',false);
    led.setDigit(0,4,mo10,false);
    led.setDigit(0,3,mo1,false);
    led.setChar (0,2,'-',false);
    led.setDigit(0,1,d10,false);
    led.setDigit(0,0,d1,false);    
  }
  #endif
  #ifdef DISPLAY_4_DIGIT
  if (s1 < 7) { // display hour,minute,second
    led.setDigit(0,3,h10,false);
    led.setDigit(0,2,h1,false);
    //if (hr12 && isPM())
    led.setDigit(0,1,m10,true);
    led.setDigit(0,0,m1,true);
    nonblockingDelay(500); // flash the dot
    led.setDigit(0,1,m10,false);
    led.setDigit(0,0,m1,false);
  } else { // display weekday,month,day
    led.setChar(0,3,w10[w],false);
    led.setChar(0,2,w1[w],false);
    led.setDigit(0,1,d10,false);
    led.setDigit(0,0,d1,true);    
  }
  #endif
}


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);

  //// MAX7219 LED display stuff
  led.shutdown(0,false);  // The MAX72XX is in power-saving mode on startup
  led.setIntensity(0,brightness);  // set the brightness
  #if defined(DISPLAY_8_DIGIT)
    led.setChar(0,7,'C',false);
    led.setChar(0,6,'o',false);
    led.setChar(0,5,'n',false);
    led.setChar(0,4,'n',false);
    led.setChar(0,3,'-',false);
    led.setChar(0,2,'-',false);
    led.setChar(0,1,'-',false);
    led.setChar(0,0,'-',false);
  #endif
  #if defined(DISPLAY_4_DIGIT)
    led.setChar(0,3,'C',false);
    led.setChar(0,2,'o',false);
    led.setChar(0,1,'n',false);
    led.setChar(0,0,'n',false);
  #endif

  //// WiFi stuff
  setupAP();  // setup access point using default ssid "WiFi setup ESP8266-xxxx" and no password
//  char devicename[] = "ESP8266 LED clock"; // optional. change to whatever you want. Use "ping ESP8266" to test
  char devicename[] = "ESP8266 4x7-segment led clock"; // optional. change to whatever you want. Use "ping ESP8266" to test
  if (setupWiFi(devicename, false)) {  // no blocking
    setClock(PST);
    digitalWrite(statusLed, HIGH);  // turn off led
  } else {
  #if defined(DISPLAY_8_DIGIT)
    led.setChar(0,7,'F',false);
    led.setChar(0,6,'A',false);
    led.setChar(0,5,'I',false);
    led.setChar(0,4,'L',false);
    led.setChar(0,3,' ',false);
    led.setChar(0,2,' ',false);
    led.setChar(0,1,' ',false);
    led.setChar(0,0,' ',false);
  #endif
  #if defined(DISPLAY_4_DIGIT)
    led.setChar(0,3,'F',false);
    led.setChar(0,2,'A',false);
    led.setChar(0,1,'I',false);
    led.setChar(0,0,'L',false);
  #endif
    // block for 3 minutes
    unsigned long timeout = millis() + 180000; // 3 minutes
    while(millis() < timeout) {
      flashActivityLed(3000,150,1);
      server.handleClient();
      yield();  // must have this to give time to handle wifi background stuff
    }
    ESP.restart();  // have not connected to the internet in 3 minutes so just reboot
  }

  // update systemStartTime and systemStartCount
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();

}

void loop() {
  static time_t prevDisplay;
  if (now() != prevDisplay) { //update the display only if time has changed
    prevDisplay = now();
    displayClock();
//    Serial.println(formattedTime());
  }
  
  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  flashActivityLed();
  // checkWifiConnection();  // reboot if wifi is disconnected
  syncClock(); // need to call this frequently to sync the clock
  server.handleClient();
  // yield();  // don't need this because yield() is called automatically after the loop() function exits
}
