/* //////////////////////////////////////////////////////////
 * A Basic Web Server using the ESP8266 or ESP32
 * to control the built-in LED
 *
 * After uploading this program to the ESP module,
 * open the serial monitor and restart the ESP module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address. For example
 *     192.168.1.99            for the main webpage
 *   or
 *     192.168.1.99/setup      for the WiFi setup page
 * 
 * or
 * 
 * Method 2 for after the program is up and running.
 * Connect your WiFi to this access point with the default ssid 
 *     "WiFi setup ESP-xxxx"
 * or the one you provided, then browse to the IP address
 *     1.2.3.4                  for the main webpage
 *   or
 *     1.2.3.4/setup            for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to
 * use method 2 above to connect your WiFi to the access point
 * and then browse to 1.2.3.4/setup to setup the ssid and password.
 * Alternatively, you can specify the ssid and password to connect to
 * in the setupWiFi(ssid, password) function call.
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the built-in led on pin 1
 * 
 * Copyright (c) 2016, 2020 Enoch Hwang
 */

// The two tokens ESP32 and ESP8266 are automatically defined
// when the respective board is selected from the menu Tools/Board
//#define ESP8266_01  // define this when using the ESP8266-01
                      
//////////////////////////////////////////////////////////////
//// VCC voltage stuff
#ifndef ESP32
  ADC_MODE(ADC_VCC);
#endif

//////////////////////////////////////////////////////////////
//// status LED stuff
#ifdef ESP8266_01
  #define statusLed 1     // for ESP8266-01
#else
  #define statusLed 2     // for ESP8266-12 and ESP32
#endif

//////////////////////////////////////////////////////////////
//// NTP clock stuff
#include "DateTime_RobotsForFun.h"

//////////////////////////////////////////////////////////////
//// WiFi stuff
//#define SERVERPORT 1040        // set server port. Default is 80
//#define USE_DEFAULT_INDEXHTML  // define this to use the default indexHTML function
                                 // otherwise create your own indexHTML below
#include "WiFi_RobotsForFun.h"

//////////////////////////////////////////////////////////////
//// HTML webpage
#ifndef USE_DEFAULT_INDEXHTML
// this routine is called when a client browse to the root IP address
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("LED")) {
    if (String(server.arg("LED")) == "OFF") {
      digitalWrite(statusLed, LEDOFF);  // turn off LED
      ledStatus = LEDOFF;
      Serial.println("turn off LED");
    } else if (String(server.arg("LED")) == "ON") {
      digitalWrite(statusLed, LEDON);   // turn on LED
      ledStatus = LEDON;
      Serial.println("turn on LED");
    } else if (String(server.arg("LED")) == "BLINK") {
      ledStatus = LEDBLINK;                // blink LED
      Serial.println("blink LED");
    }
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  //msg += "<meta http-equiv=\"refresh\" content=\"3\">"; // auto refresh in 3 seconds
  #ifdef ESP32
    msg += "<title>ESP32 Demo</title>";
  #else
    #ifdef ESP8266_01
      msg += "<title>ESP8266-01 Demo</title>";
    #else
      msg += "<title>ESP8266-12 Demo</title>";
    #endif
  #endif
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  #ifdef ESP32
    msg += "<center><font size = \"3\"><b>ESP32 Demo Web Server</b></font><br>";
    msg += "This is an ESP32 demo webserver that has been running for ";
  #else
    msg += "<center><font size = \"3\"><b>ESP8266 Demo Web Server</b></font><br>";
    #ifdef ESP8266_01
      msg += "This is an ESP8266-01 demo webserver that has been running for ";
    #else
      msg += "This is an ESP8266-12 demo webserver that has been running for ";
    #endif
  #endif
  msg += millis()/1000;
  msg += " seconds.";
  msg += "<br>You can control the built-in LED with this webpage.";
  #ifdef __DateTime_RobotsForFun__
    msg += "<br>" + formattedTime();
  #endif
  msg += "</center>";
  msg += "<hr><br>";

  msg += "Current led status is ";
  switch(ledStatus) {
    case LEDOFF: 
      msg += "<b>OFF</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=ON\"><button style=\"font-size:16pt\">ON</button></a>";
      break;
    case LEDON: 
      msg += "<b>ON</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=BLINK\"><button style=\"font-size:16pt\">BLINK</button></a>";
      break;
    default: 
      msg += "<b>BLINK</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a>";
      break;
  }

  // another way to create the button
  msg += "<br><br>Another way to do it:";
  msg += "<form style=\"font-size:12pt\">";
  msg += "Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn off led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn on led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"BLINK\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to blink led";
  msg += "</form>";

  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }
  if (internetIsConnected) { // Connect to DNS server to get IP address
    msg += "<br>Go to <a href=\"http://robotsforfun.com\">RobotsForFun.com</a> for more interesting projects.<br>";
    msg += "<center><a href=\"http://robotsforfun.com\"><img width=\"100\" src=\"http://robotsforfun.com/images/Robo.png\" alt=\"Robo the robot\"></a></center>";
  }
  msg += "</body>";
  msg += "</html>";
  // end of webpage file
  
  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}
#endif


void setup() {
  Serial.begin(115200);
  pinMode(statusLed, OUTPUT);
  delay(500);
  #ifdef ESP32
    Serial.println("ESP32 chip");
  #endif
  #ifdef ESP8266
    Serial.println("ESP8266 chip");
  #endif
    
  //// set clock time zone to PST
  setClock(PST);
  
  //// WiFi stuff
  char devicename[] = "ESP Demo Webserver"; // optional. change to whatever you want. Use "ping ESP8266" to test
  char ssid[] = "your_ssid";           // change this to your ssid
  char password[] = "your_password";   // change this to your password

  // optional to setup an access point (AP) for devices to connect to this ESP directly without needing a wifi router
  setupAP(devicename);  // setup access point with the devicename as the ssid and with no password
  // setupAP();  // setup access point with the default ssid "WiFi setup ESP-xxxx" and with no password

  // The following are different ways to connect this ESP to a given wifi router
  // It's better not to set the ssid and password in the setupWiFi() call.
  // Instead connect to the AP and set the ssid and password in the setup page at 1.2.3.4/setup after the AP server is running

  bool connected;
  
  // use DHCP, given device name, given ssid and password, and default blocking if cannot connect to wifi
  connected = setupWiFi(devicename, ssid, password);

  // use DHCP, default device name, ssid and password obtained from EEPROM, and default blocking if cannot connect to wifi
  // connected = setupWiFi();

  // use DHCP, given devicename, ssid and password obtained from EEPROM, and no blocking if cannot connect to wifi
  // connected = setupWiFi(devicename, false);

  // use DHCP, default device name, given ssid and password, and default blocking if cannot connect to wifi
  // connected = setupWiFi(ssid, password);

  // use given device name, ssid and password obtained from EEPROM, given static IP, and blocking if cannot connect to wifi
  // IPAddress staticIP(192, 168, 1, 99); // use this if you want a static IP address
  // connected = setupWiFi(devicename, staticIP, true);

  // use given device name, given ssid and password, staticIP, and blocking if cannot connect to wifi
  // connected = setupWiFi(devicename, ssid, password, staticIP, true);

  // first try to connect to the ssid and password obtained from EEPROM
  // if unsuccessful then try to connect to the ssids and passwords passed in
  // const int NUMBER_OF_SSIDS = 4;
  // const char default_ssids[NUMBER_OF_SSIDS][SSID_SIZE] = {"ssid_1", "ssid_2", "ssid_3", "ssid_4"};
  // const char default_passwords[NUMBER_OF_SSIDS][PASSWORD_SIZE] = {"password_1", "password_2", "password_3", "password_4"};
  // connected = setupWiFi(devicename, default_ssids, default_passwords, NUMBER_OF_SSIDS);  // no blocking

  // wait for clock to sync with server
  #ifdef __DateTime_RobotsForFun__
    if (internetIsConnected) {
      if (waitForTimeValid()) {
        Serial.print("Current time is ");
        Serial.println(formattedTime());
      } else {
        Serial.println("Failed to connect to NTP server");
      }
    }
  #endif
                 
  // update systemStartTime and systemStartCount. This is for information purposes only
  systemStartTime = now();
  EEPROM.get(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // get systemStartCount from EEPROM
  systemStartCount++;
  EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);
  EEPROM.commit();
  EEPROM.get(EEPROM_ADDRESS_ledStatus, ledStatus);  // get led status  
}


void loop() {
//   nonblockingDelay(1000);           // use this delay to handle web clients
//   Serial.println(formattedTime());  // print out current time

  if (APisOn && wifiIsConnected && (millis() > 600000)) turnOffAP(); // turn off AP after 10 minutes
  if (wifiIsConnected) {
    flashActivityLed();     // double flash to show system WiFi is connected and system is alive
  } else {
    flashActivityLed(2000, 150, 1);     // slow flash to show WiFi not connected
  }
  server.handleClient();  // handle web client request
  yield();  // yield() is automatically called at the end of loop()
            // otherwise must have this in a loop to give time to handle wifi background stuff
}
