/*
 * Library routines to connect the ESP8266 WiFi module 
 * and start a webserver on it.
 * 
 * After uploading this program to the ESP8266-12 module,
 * open the serial monitor and restart the ESP8266 module to 
 * see what IP address is assigned to this webserver,
 * then browse to this IP address. For example
 *     192.168.1.99            for the main webpage
 *   or
 *     192.168.1.99/setup      for the WiFi setup page
 * 
 * or
 * 
 * Method 2 for after the program is up and running.
 * Connect your WiFi to this access point with the default ssid 
 *     "WiFi setup ESP8266-xxxx"
 * or the one you provided, then browse to the IP address
 *     1.2.3.4                  for the main webpage
 *   or
 *     1.2.3.4/setup            for the WiFi setup page
 * 
 * If the WiFi is not connected then you will have to
 * use method 2 above to connect your WiFi to the access point
 * and then browse to 1.2.3.4/setup to setup the ssid and password.
 * Alternatively, you can specify the ssid and password to connect to
 * in the setupWiFi(ssid, password) function call.
 * 
 * If using the ESP8266-01 module, nothing will be printed
 * on the serial monitor because it conflicts with 
 * the built-in led on pin 1
 * 
 * 
 * Copyright (c) 2020 Enoch Hwang
 */

#ifndef __WiFi_RobotsForFun__
#define __WiFi_RobotsForFun__
 
//////////////////////////////////////////////////////////////
//// Wifi library stuff
#ifdef ESP8266
  #include <ESP8266WiFi.h>
  #include <ESP8266WebServer.h>
  #ifndef SERVERPORT
    #define SERVERPORT 80 // default server port for http
  #endif
  #define ARDUINO_EVENT_WIFI_STA_CONNECTED 0
  #define ARDUINO_EVENT_WIFI_STA_DISCONNECTED 1
  #define ARDUINO_EVENT_WIFI_STA_AUTHMODE_CHANGE 2
  #define ARDUINO_EVENT_WIFI_STA_GOT_IP 3
  ESP8266WebServer server(SERVERPORT);	// create a webserver object to listen on server port

  //////////////////////////////////////////////////////////////
  //// OTA (Over The Air) updater stuff
  // * To update a new firmware, browse to this webserver's
  // * IP address /update. e.g.
  // *     1.2.3.4/update       for the update page
  // * From the update page, choose the .bin file to upload    
  // * e.g. BasicWebServer.ino.bin
  // * This bin file should be located in
  // * c:/Users/<account>/AppData/Local/Temp/arduino_build_xxxxxx
  #include <ESP8266HTTPUpdateServer.h>
  ESP8266HTTPUpdateServer httpUpdater;
#endif

#ifdef ESP32 
  #include <WiFi.h>
  #include <WebServer.h>
  #ifndef SERVERPORT
    #define SERVERPORT 80 // default server port for http
  #endif
  // the event tokens defined for the ESP8266 are already defined for the ESP32
  WebServer server(SERVERPORT);	// create a webserver object to listen on server port
  #include <Update.h>           // OTA (Over The Air) updater
#endif


//////////////////////////////////////////////////////////////
//// system start info stuff
#define SSID_SIZE 31
#define PASSWORD_SIZE 31
char _ssid[SSID_SIZE];
char _password[PASSWORD_SIZE];
char _APssid[SSID_SIZE];
char _APpassword[PASSWORD_SIZE];
// the values in these two variables must be updated in your program to be correct
unsigned long systemStartTime = 0;  // time system started
unsigned int systemStartCount = 0;  // how many times system started
bool wifiIsConnected = false;       // flag for whether WiFi is connected or not
bool internetIsConnected = false;   // flag for whether internet is connected or not
bool APisOn = false;                // flag for whether AP is on or not
//int disconnectedCount = 0;          // for early termination of invalid SSID

//////////////////////////////////////////////////////////////
//// status LED stuff
#ifndef statusLed
  #ifdef ESP8266_01
    #define statusLed 1     // built-in led for ESP8266-01
  #else
    #define statusLed 2     // built-in led for ESP8266-12
  #endif
#endif

#ifdef ESP32
  #define LEDOFF   0  // turn off led with a 0 for ESP32
  #define LEDON    1
  #define LEDBLINK 2
#else
  #define LEDOFF   1  // turn off led with a 1 for ESP8266
  #define LEDON    0
  #define LEDBLINK 2
#endif  
unsigned int ledStatus = LEDBLINK;  // 0=off, 1=on, 2=blink

// Flashes the led without using the delay() function.
// The period of each cycle is flashDelay milliseconds
// flashSpeed determines the speed of each flash
// flashTimes determines the number of times to flash in a cycle
// This version needs to be called inside a loop in order to flash
// In order to notice a pause between each cycle
// flashDelay should be greater than 2*flashSpeed*flashTimes
//  e.g.   flashActivityLed(1000, 100, 1);
void flashActivityLed(int flashDelay=3000, int flashSpeed=150, int flashTimes=2, bool resetTime=false) {
  static unsigned long nextLedActivity = 0;
  static int t = 0;
  if (ledStatus != LEDBLINK) {
    digitalWrite(statusLed, ledStatus); // turn on or off LED
    return;  // return if ledStatus is 0=off or 1=on
  }
  // blink the LED
  if (resetTime) {
    nextLedActivity = 0; // reset next flash time
    digitalWrite(statusLed, LEDOFF); // turn off LED
  }
  if ((millis() > nextLedActivity)) {
    if (t == 0) {
      digitalWrite(statusLed, LEDON); // turn on LED
    } else {
      digitalWrite(statusLed, !digitalRead(statusLed)); // toggle LED
    }
    if (t < flashTimes*2-1) {
      nextLedActivity = millis() + flashSpeed;
      t++;
    } else {
      nextLedActivity = millis() + flashDelay - (flashSpeed*2*flashTimes);
      t = 0;
    }
  }
}

// Do delay without using the delay() function
// wait = ms to delay
// flashLed=true to flash the status led while delaying
// Usage: nonblockingDelay(1000);
void nonblockingDelay(int wait, bool flashLed=false, int flashDelay=3000, int flashSpeed=150, int flashTimes=2, bool resetTime=false) {
  //  unsigned long timesUp;		// this will have time rollover problem
  //  timesUp = millis() + wait;
  //  while (millis() < timesUp) {
  unsigned long start = millis();
  while (millis() - start < wait) {	// compare durations will not have time rollover problem
    if (flashLed) flashActivityLed(flashDelay, flashSpeed, flashTimes, resetTime);
    server.handleClient();
    yield();  // must have this to give time to handle wifi background stuff
  }
}


//////////////////////////////////////////////////////////////
//// EEPROM stuff
#define EEPROM_ADDRESS_ssid 0
#define EEPROM_ADDRESS_password EEPROM_ADDRESS_ssid+sizeof(_ssid)
#define EEPROM_ADDRESS_ok EEPROM_ADDRESS_password+sizeof(_password)
#define EEPROM_ADDRESS_systemStartCount EEPROM_ADDRESS_ok+3
#define EEPROM_ADDRESS_pushNotification EEPROM_ADDRESS_systemStartCount+sizeof(int)
#define EEPROM_ADDRESS_ledStatus EEPROM_ADDRESS_pushNotification+sizeof(bool)
#define EEPROM_START_OF_DATA EEPROM_ADDRESS_ledStatus+sizeof(int)
#include <EEPROM.h>

bool loadWifiCredentials() {
  EEPROM.begin(512);
  EEPROM.get(EEPROM_ADDRESS_ssid, _ssid);
  EEPROM.get(EEPROM_ADDRESS_password, _password);
  char ok[2+1];
  EEPROM.get(EEPROM_ADDRESS_ok, ok);
  // EEPROM.end();
  if (String(ok) != String("OK")) {
    _ssid[0] = 0;
    _password[0] = 0;
    return false;
  } else {
    return true;
  }
}

void saveWifiCredentials() {
  EEPROM.begin(512);
  EEPROM.put(EEPROM_ADDRESS_ssid, _ssid);
  EEPROM.put(EEPROM_ADDRESS_password, _password);
  char ok[2+1] = "OK";
  EEPROM.put(EEPROM_ADDRESS_ok, ok);
  EEPROM.commit();
  // EEPROM.end();
}

// call this to erase the wifi credentials in the EEPROM
void clearWifiCredentials() {
  EEPROM.begin(512);
  _ssid[0] = 0;
  _password[0] = 0;
  EEPROM.put(EEPROM_ADDRESS_ssid, _ssid);
  EEPROM.put(EEPROM_ADDRESS_password, _password);
  char ok[2+1] = "  ";
  EEPROM.put(EEPROM_ADDRESS_ok, ok);
  EEPROM.commit();
}

//////////////////////////////////////////////////////////////
//// Wifi stuff

// returns true if wifi and intenet are connected
bool checkWifiConnection(bool reboot=true) {
  wifiIsConnected = false;
  internetIsConnected = false;
  if (WiFi.status() == WL_CONNECTED) {
    wifiIsConnected = true;
    // wifi is connected to local access point
    // but it is possible that the access point is not connected to the internet
    // so see if we can reach the internet by querying a DNS server
    IPAddress ipaddress;
    if (WiFi.hostByName("www.google.com", ipaddress) == 1) { // Connect to DNS server to get IP address
    // if (WiFi.ping("www.google.com") >= 0) {
	  internetIsConnected = true;
      return true;
    } else {
      Serial.print("*** Disconnected from the internet.  ");
      // WiFi.status() == 3
    }
  } else {
    Serial.print("*** Disconnected from the WiFi.  ");
      // WiFi.status() == 6
  }

  if (reboot) {
    Serial.print("Status=");
    Serial.print(WiFi.status());
    Serial.println(".  Rebooting in 3 minutes ***");
    unsigned long timeout = millis() + 180000; // 3 minutes
    while(millis() < timeout) {
      flashActivityLed(2000,150,1);
      server.handleClient();
      yield();  // must have this to give time to handle wifi background stuff
    }
    ESP.restart();  // have not connected to the internet in 3 minutes so just reboot
    return false;
  } else {
    return false;
  }
}

//bool wifiConnected() {
//  return wifiIsConnected;
  // using the test below is incorrect if the configTime() call is successful
  // even though the wifi is not connected
  // so need to set the wifiIsConnected flag right after connecting to the wifi
  // and before calling configTime()
  // if (WiFi.status() == WL_CONNECTED) {
    // return true;
  // } else {
    // return false;
  // }
//}

// returns the WiFi signal strength in percentage
int signalStrength() {
  long rssi = WiFi.RSSI();  // in dBm
  int percentage;
  // convert dBm to percentage
  // https://stackoverflow.com/questions/15797920/how-to-convert-wifi-signal-strength-from-quality-percent-to-rssi-dbm
  if (rssi > -50) {
    percentage = 100;
  } else if (rssi > -55) {
    percentage = 90;
  } else if (rssi > -62) {
    percentage = 80;
  } else if (rssi > -65) {
    percentage = 75;
  } else if (rssi > -68) {
    percentage = 70;
  } else if (rssi > -74) {
    percentage = 60;
  } else if (rssi > -79) {
    percentage = 50;
  } else if (rssi > -83) {
    percentage = 30;
  }
  return percentage;  
}



//////////////////////////////////////////////////////////////
//// udp stuff
#include <WiFiUdp.h>
#ifndef UDPreceiverPort
  #define UDPreceiverPort 1070
#endif
WiFiUDP udp;


#ifdef ESP32 
  // webpage for inputting the filename for OTA upload
  const char* inputFileNameHTML =
    "Select bin file to upload. On Windows' machines, it is located in<br>"
    "&nbsp;&nbsp;user name/AppData/Local/Temp/arduino_build_*/project name.ino.bin<br><br>"
    "<script src='https://ajax.googleapis.com/ajax/libs/jquery/3.2.1/jquery.min.js'></script>"
    "<form method='POST' action='#' enctype='multipart/form-data' id='upload_form'>"
       "<input type='file' name='update'>"
            "<input type='submit' value='Update'>"
        "</form>"
     "<div id='prg'>progress: 0%</div>"
     "<script>"
      "$('form').submit(function(e){"
      "e.preventDefault();"
      "var form = $('#upload_form')[0];"
      "var data = new FormData(form);"
      " $.ajax({"
      "url: '/update',"
      "type: 'POST',"
      "data: data,"
      "contentType: false,"
      "processData:false,"
      "xhr: function() {"
      "var xhr = new window.XMLHttpRequest();"
      "xhr.upload.addEventListener('progress', function(evt) {"
      "if (evt.lengthComputable) {"
      "var per = evt.loaded / evt.total;"
      "$('#prg').html('progress: ' + Math.round(per*100) + '%');"
      "}"
      "}, false);"
      "return xhr;"
      "},"
      "success:function(d, s) {"
      "console.log('success!')"
     "},"
     "error: function (a, b, c) {"
     "}"
     "});"
     "});"
     "</script>";

  // webpage to automatically refresh in 3 seconds
  const char* refreshHTML =
    "<!DOCTYPE html>"
    "<head>"
    "<meta http-equiv=\"refresh\" content=\"3\">"
    "</head>"
    "<body>"
    "Page is going to refresh in 3 seconds..."
    "</body>"
    "</html>";
     
#endif

//////////////////////////////////////////////////////////////
//// Station STA stuff
// forward declarations
void indexHTML();
void setupHTML();
void notfoundHTML();


////////////////////////
// WiFi events handler
// This is automatically called when there's a wifi event
// https://github.com/espressif/arduino-esp32/blob/master/libraries/WiFi/examples/WiFiClientEvents/WiFiClientEvents.ino
void WiFiEventsHandler(WiFiEvent_t event) {
  //Serial.printf("[WiFi-event] event: %d\n", event);
  switch (event) {
    case ARDUINO_EVENT_WIFI_STA_CONNECTED:
    //case 0:   // for ESP8266
    //case 112: // for ESP32
      // at this point WiFi.status() != WL_CONNECTED
      //Serial.println("Connected to WiFi");
      break;

    case ARDUINO_EVENT_WIFI_STA_DISCONNECTED:
    //case 1:   // for ESP8266
    //case 113: // for ESP32
      //disconnectedCount++;  // disconnectedCount > 1 in about 2 seconds if SSID is invalid
      //Serial.println("Disconnected from WiFi");
      break;

    case ARDUINO_EVENT_WIFI_STA_AUTHMODE_CHANGE:
    //case 2:   // for ESP8266
    //case 114: // for ESP32
      //Serial.println("Auth Mode Change");
      break;
    
    case ARDUINO_EVENT_WIFI_STA_GOT_IP:
    //case 3:   // for ESP8266
    //case 115: // for ESP32
      // at this point WiFi.status() == WL_CONNECTED
      //disconnectedCount = 0;
      //Serial.print("Obtained IP address: ");
      //Serial.println(WiFi.localIP());
      break;
    default:
      //Serial.printf("[WiFi-event] event: %d\n", event);
      break;
  }
}


////////////////////////
// calling this will use DHCP to get IP address
// returns true if connection is successful, otherwise returns false
// if blocking is true then this function will NOT return if cannot connect to wifi
// original bool setupWiFi(const char devicename[]="", bool blocking=true) {  
bool setupWiFi(const char devicename[], const char ssid[], const char password[], bool blocking=true) {
  wifiIsConnected = false;
  internetIsConnected = false;
	if (strlen(devicename) > 0) {
		#ifdef ESP32
			WiFi.setHostname(devicename); 	// set the device name
		#else
			WiFi.hostname(devicename);	    // set the device name
		#endif
	}
	
	Serial.print("\nConnecting WiFi to [");
	Serial.print(ssid);
	Serial.print("] using password [");
	Serial.print(password);
	Serial.println("]");
  
  //WiFi.disconnect();  
  WiFi.onEvent(WiFiEventsHandler);  // listen to wifi events.
  WiFi.begin(ssid, password);
  unsigned long startTime = millis();
  while ((WiFi.status() != WL_CONNECTED) && millis() - startTime < 30000) {	// 30 seconds timeout
    if (WiFi.status() == WL_NO_SSID_AVAIL) {  // invalid SSID
     Serial.println("WL_NO_SSID_AVAIL");
     break;  // for early termination if invalid SSID
    }
		digitalWrite(statusLed, !digitalRead(statusLed));
		delay(125);
		Serial.print(".");
	}
  
	Serial.println();
	if (WiFi.status() == WL_CONNECTED) {	// wifi connected
		wifiIsConnected = true;
		digitalWrite(statusLed, LEDON);
    
    // save ssid and password to EEPROM
    if (strlen(ssid) > 0) {
      stpcpy(_ssid, ssid);
      stpcpy(_password, password);
      saveWifiCredentials();
    }

    if (APisOn) {
      WiFi.mode(WIFI_AP_STA);	// use both AP and station modes
    } else {
      WiFi.mode(WIFI_STA);	  // use just station mode
    }
    
    // start the webserver
    server.on("/", indexHTML);
    server.on("/setup", setupHTML);
	  server.onNotFound(notfoundHTML);
	  server.begin();
	  
	  // UDP stuff
	  udp.begin(UDPreceiverPort);
	  
	  // start the OTA updater server
	  #ifdef ESP32 // OTA for ESP32
		server.on("/updateCalled", HTTP_GET, []() {
		  server.sendHeader("Connection", "close");
		  server.send(200, "text/html", inputFileNameHTML);
		});
		
		server.on("/update", HTTP_POST, []() {
		  server.sendHeader("Connection", "close");
		  server.send(200, "text/plain", (Update.hasError()) ? "FAIL" : "OK");
		  ESP.restart();
		}, []() {
		  HTTPUpload& upload = server.upload();
		  if (upload.status == UPLOAD_FILE_START) {
			Serial.printf("Update: %s\n", upload.filename.c_str());
			if (!Update.begin(UPDATE_SIZE_UNKNOWN)) { //start with max available size
			  Update.printError(Serial);
			}
		  } else if (upload.status == UPLOAD_FILE_WRITE) {
			/* flashing firmware to ESP*/
			if (Update.write(upload.buf, upload.currentSize) != upload.currentSize) {
			  Update.printError(Serial);
			}
		  } else if (upload.status == UPLOAD_FILE_END) {
			if (Update.end(true)) { //true to set the size to the current progress
			  Serial.printf("Update Success: %u\nRebooting...\n", upload.totalSize);
			} else {
			  Update.printError(Serial);
			}
		  }
		});

	  #else // OTA for ESP8266
		httpUpdater.setup(&server);
	  #endif
	  
	  Serial.println();
	  Serial.println("WiFi connected");
	  Serial.print("  IP address: ");
	  Serial.println(WiFi.localIP());
	  Serial.print("  Port:       ");
	  Serial.println(SERVERPORT);
	  Serial.print("  Gateway:    ");
	  Serial.println(WiFi.gatewayIP().toString());
	  Serial.print("  Hostname:   ");
	  #ifdef ESP32
		Serial.println(WiFi.getHostname());
	  #else
		Serial.println(WiFi.hostname());
	  #endif
	  // print the received signal strength:
	  long rssi = WiFi.RSSI();  // in dBm
	  Serial.print("  Signal strength: ");
	  Serial.print("RSSI ");
	  Serial.print(rssi);
	  Serial.print(" dBm ");
	  Serial.print(signalStrength());
	  Serial.println("%");
	  digitalWrite(statusLed, LEDOFF);
	  // check if there's internet connection
	  IPAddress ipaddress;
	  if (WiFi.hostByName("www.google.com", ipaddress) == 1) { // Connect to DNS server to get IP address
      Serial.println("Internet connected");
      internetIsConnected = true;
      return true;
    } else {
      Serial.println("No internet connection");
      internetIsConnected = false;
      return false;
    }

  } else {	// wifi not connected
    digitalWrite(statusLed, LEDOFF);
    wifiIsConnected = false;
    Serial.println("\nWiFi is not connected");
    if (APisOn) {
      Serial.print("  Connect your WiFi to [");
      Serial.print(_APssid);
      Serial.print("]");
      if (strlen(_APpassword) > 0) {
        Serial.print(" using password [");
        Serial.print(_APpassword);
        Serial.print("]");
      }
      Serial.println();
      Serial.print("  and then Browse to [");
      Serial.print(WiFi.softAPIP());
      Serial.println("/setup] to setup WiFi");
    } else {
      Serial.println("and AP is not setup");
    }
    if (blocking) {
      Serial.print("*** Status=");
      Serial.print(WiFi.status());  // WiFi.status() == 1
      Serial.println(".  Rebooting in 3 minutes ***");
      unsigned long start = millis();
      while (millis() - start < 180000) { // 3 minutes. compare duration will not have rollover problem
        flashActivityLed(2000,150,1);
        server.handleClient();
        yield();  // must have this to give time to handle wifi background stuff
      }
      ESP.restart();  // have not connected to the internet in 3 minutes so just reboot
    }
    return false;
  }
}


//////////////////////// done
// calling this will use the given ssid, password, and blocking
bool setupWiFi(const char ssid[], const char password[], bool blocking=true) {
  // if (strlen(ssid) > 0) {
    // stpcpy(_ssid, ssid);
    // stpcpy(_password, password);
    // saveWifiCredentials();
  // }
	// return setupWiFi("", blocking);
  return setupWiFi("", ssid, password, blocking);
}

//////////////////////// done
// calling this will use DHCP to get IP address with an optional device name
// the ssid and password are obtained from the EEPROM
// returns true if connection is successful, otherwise returns false
bool setupWiFi(const char devicename[]="", bool blocking=true) {  
  if (loadWifiCredentials()) {
	  return setupWiFi(devicename, _ssid, _password, blocking);
  } else {
    digitalWrite(statusLed, LEDOFF);
    Serial.println();
    Serial.println("Invalid ssid and/or password loaded from EEPROM");
	return false;
  }
}



//////////////////////// done
// calling this will use the given devicename, ssid, password, staticIP, and blocking
bool setupWiFi(const char devicename[], const char ssid[], const char password[], IPAddress staticIP, bool blocking=true) {
  // if (strlen(ssid) > 0) {
    // stpcpy(_ssid, ssid);
    // stpcpy(_password, password);
    // saveWifiCredentials();
  // }
  
  // setup static IP address
  IPAddress subnet(255, 255, 255, 0);
  // the first 3 octets for the gateway and dns should be the same as the IP
  IPAddress gateway(192, 168, 1, 1);
  IPAddress dns(192, 168, 1, 1);
  gateway[0] = dns[0] = staticIP[0];
  gateway[1] = dns[1] = staticIP[1];
  gateway[2] = dns[2] = staticIP[2];
  WiFi.config(staticIP, gateway, subnet, dns);          // must have dns, otherwise cannot connect to NTP server

  return setupWiFi(devicename, ssid, password, blocking);
}

//////////////////////// done
// calling this will use the given ssid, password, staticIP, and blocking
bool setupWiFi(const char ssid[], const char password[], IPAddress staticIP, bool blocking=true) {
  // if (strlen(ssid) > 0) {
    // stpcpy(_ssid, ssid);
    // stpcpy(_password, password);
    // saveWifiCredentials();
  // }
  
  // setup static IP address
  // IPAddress subnet(255, 255, 255, 0);
  // the first 3 octets for the gateway and dns should be the same as the IP
  // IPAddress gateway(192, 168, 1, 1);
  // IPAddress dns(192, 168, 1, 1);
  // gateway[0] = dns[0] = staticIP[0];
  // gateway[1] = dns[1] = staticIP[1];
  // gateway[2] = dns[2] = staticIP[2];
  // WiFi.config(staticIP, gateway, subnet, dns);          // must have dns, otherwise cannot connect to NTP server

  // return setupWiFi("", blocking);
  return setupWiFi("", ssid, password, staticIP, blocking);
}


//////////////////////// done
// calling this will use the given device name and static IP address
// the ssid and password are obtained from the EEPROM
// if blocking is true then this function will NOT return if cannot connect to wifi
bool setupWiFi(const char devicename[], IPAddress staticIP, bool blocking=true) {
  // IPAddress subnet(255, 255, 255, 0);
  // the first 3 octets for the gateway and dns should be the same as the IP
  // IPAddress gateway(192, 168, 1, 1);
  // IPAddress dns(192, 168, 1, 1);
  // gateway[0] = dns[0] = staticIP[0];
  // gateway[1] = dns[1] = staticIP[1];
  // gateway[2] = dns[2] = staticIP[2];

  // WiFi.config(staticIP, gateway, subnet, dns);          // must have dns, otherwise cannot connect to NTP server
  // return setupWiFi(devicename, blocking);

  if (loadWifiCredentials()) {
    return setupWiFi(devicename, _ssid, _password, staticIP, blocking);
  } else {
    digitalWrite(statusLed, LEDOFF);
    Serial.println();
    Serial.println("Invalid ssid and/or password loaded from EEPROM");
    return false;
  }  
}


////////////////////////
// calling this will use the given device name, static IP address and gateway
bool setupWiFi(const char devicename[], IPAddress staticIP, IPAddress gateway, bool blocking=true) {
  if (loadWifiCredentials()) {
    IPAddress subnet(255, 255, 255, 0);
    // dns is the same as the gateway
    IPAddress dns(192, 168, 1, 1);
    dns[0] = gateway[0];
    dns[1] = gateway[1];
    dns[2] = gateway[2];
    dns[3] = gateway[3];
    WiFi.config(staticIP, gateway, subnet, dns);          // must have dns, otherwise cannot connect to NTP server

    return setupWiFi(devicename, _ssid, _password, blocking);
//  return setupWiFi(devicename, blocking);
  } else {
    digitalWrite(statusLed, LEDOFF);
    Serial.println();
    Serial.println("Invalid ssid and/or password loaded from EEPROM");
    return false;
  }
}


////////////////////////
// calling this will first use the ssid and password obtained from the EEPROM
// if it fails then it will try to connect to each of the ssids passed in
// The caller must set up the following two arrays with the ssids and passwords to pass in
// const char default_ssids[number_of_ssids][SSID_SIZE] = {"ssid_1", "ssid_2", "ssid_3", "ssid_4"};
// const char default_passwords[number_of_ssids][PASSWORD_SIZE] = {"password_1", "password_2", "password_3", "password_4"};
bool setupWiFi(const char devicename[], 
	const char ssids[][SSID_SIZE], 
	const char passwords[][PASSWORD_SIZE], const int number_of_ssids) {
		
  bool connected = setupWiFi(devicename, false);   // use DHCP and no blocking

  // Don't scan network. Just try to connect to each of the provided ssids
  for (int j=0; j<number_of_ssids  && !connected; j++) {
    connected = setupWiFi(devicename, ssids[j], passwords[j], false);   // use DHCP and no blocking
  }
  
  // scan available networks for one of the default ssid's
  // if (!connected) {
    // Serial.println("Scanning network");
    // int n = WiFi.scanNetworks();
    // Serial.println(" # | SSID                             | RSSI");
    // for (int i=0; i<n && !connected; i++) {
      // Serial.printf("%2d",i);
      // Serial.print(" | ");
      // Serial.printf("%-32.32s", WiFi.SSID(i).c_str());
      // Serial.print(" | ");
      // Serial.printf("%4d\n", WiFi.RSSI(i));
      // for (int j=0; j<number_of_ssids  && !connected; j++) {
        // if (WiFi.SSID(i) == ssids[j]) {  // found a default ssid
          // connected = setupWiFi(devicename, ssids[j], passwords[j], false);
        // }
      // }
    // }
    // WiFi.scanDelete();
  // }
  return connected;
}

//////////////////////////////////////////////////////////////
//// Access point AP stuff
// Reference:
//   https://arduino-esp8266.readthedocs.io/en/latest/esp8266wifi/soft-access-point-class.html
void setupAP(const char ssid[] = "WiFi setup ESP-xxxx", const char password[] = "", int channel=1, int ssid_hidden=0) {
  // channel: 1=default
  // ssid_hidden: 0=not hidden; 1=hidden

  strcpy(_APssid, ssid);
  strcpy(_APpassword, password);

  if (String(_APssid) == "WiFi setup ESP-xxxx") {  // use default ssid name
    _APssid[15] = WiFi.macAddress()[12];  // replace with last 4 hex mac address
    _APssid[16] = WiFi.macAddress()[13];
    _APssid[17] = WiFi.macAddress()[15];
    _APssid[18] = WiFi.macAddress()[16];
  }
  
  // if password is "" then AP is opened, otherwise it must be at least 8 characters long
  if (strlen(_APpassword) > 0 && strlen(_APpassword) < 8) {
    _APpassword[0] = 0;
    Serial.println("AP password less than 8 characters. Reset to empty");
  }

  // optional to set ip address of AP
  IPAddress APip(1, 2, 3, 4);
  IPAddress APgateway(1, 2, 3, 4);  // the ip address and the gateway have to be the same
  IPAddress APsubnet(255, 255, 255, 0);
  WiFi.softAPConfig(APip, APgateway, APsubnet);

  // start the AP
  if (WiFi.softAP(_APssid, _APpassword)) {
    APisOn = true;
    if (wifiIsConnected) {
      WiFi.mode(WIFI_AP_STA);	// use both AP and station modes
    } else {
      WiFi.mode(WIFI_AP);	    // use just AP mode
    }

    // start the webserver
    server.on("/", indexHTML);
    server.on("/setup", setupHTML);
    // server.onNotFound(setupHTML);
    server.onNotFound(notfoundHTML);
    
    #ifdef ESP32
      server.on("/updateCalled", HTTP_GET, []() {
        server.sendHeader("Connection", "close");
        server.send(200, "text/html", inputFileNameHTML);
      });

//        server.on("/update", HTTP_GET, []() {
//          Serial.println("on update send automatic page refresh EH2");
//          server.send(200, "text/html", refreshHTML);
//        });

      server.on("/update", HTTP_POST, []() {
        server.sendHeader("Connection", "close");
        server.send(200, "text/plain", (Update.hasError()) ? "FAIL" : "OK");
        ESP.restart();
      }, []() {
        HTTPUpload& upload = server.upload();
        if (upload.status == UPLOAD_FILE_START) {
          Serial.printf("Update: %s\n", upload.filename.c_str());
          if (!Update.begin(UPDATE_SIZE_UNKNOWN)) { //start with max available size
            Update.printError(Serial);
          }
        } else if (upload.status == UPLOAD_FILE_WRITE) {
          /* flashing firmware to ESP*/
          if (Update.write(upload.buf, upload.currentSize) != upload.currentSize) {
            Update.printError(Serial);
          }
        } else if (upload.status == UPLOAD_FILE_END) {
          if (Update.end(true)) { //true to set the size to the current progress
            Serial.printf("Update Success: %u\nRebooting...\n", upload.totalSize);
          } else {
            Update.printError(Serial);
          }
        }
      });

    #endif
    server.begin();
    // UDP stuff
    udp.begin(UDPreceiverPort);
    
    Serial.println("\nAccess point started");
    Serial.print("  ssid: [");
    Serial.print(_APssid);
    Serial.println("]");
    //if (strlen(_APpassword) > 0) {
      Serial.print("  password: [");
      Serial.print(_APpassword);
      Serial.println("]");
    //}
    Serial.print("  URL:  [");
    Serial.print(WiFi.softAPIP());
    Serial.print("] or [");
    Serial.print(WiFi.softAPIP());
    Serial.println("/setup]");
  } else {
    _APssid[0] = 0;
    _APpassword[0] = 0;
    Serial.println("\nAccess point setup failed");    
  }
}

void turnOffAP() {
  // turn AP off
  APisOn = false;
  WiFi.softAPdisconnect(true);  // disconnect AP only; not STA
  WiFi.mode(WIFI_STA);	        // use just station mode
  Serial.println("Turn AP off");
}


//////////////////////////////////////////////////////////////
//// Webserver HTML webpage stuff

// stops the webserver
void stopWebserver() {
  server.stop();
}


// #define USE_DEFAULT_INDEXHTML  // define this to use the following indexHTML() function
#ifdef USE_DEFAULT_INDEXHTML
void indexHTML() {
  // to get fast responsive webpage the code needs to execute the
  //   server.send(200, "text/html", msg);
  // command in this function as soon as possible

  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("LED")) {
    if (String(server.arg("LED")) == "OFF") {
      digitalWrite(statusLed, LEDOFF);  // turn off LED
      ledStatus = LEDOFF;
      Serial.println("turn off LED");
    } else if (String(server.arg("LED")) == "ON") {
      digitalWrite(statusLed, LEDON);   // turn on LED
      ledStatus = LEDON;
      Serial.println("turn on LED");
    } else if (String(server.arg("LED")) == "BLINK") {
      ledStatus = LEDBLINK;                // blink LED
      Serial.println("blink LED");
    }
  }

  // construct the webpage file using HTML coding
  String msg = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  //msg += "<meta http-equiv=\"refresh\" content=\"3\">"; // auto refresh in 3 seconds
  #ifdef ESP32
    msg += "<title>ESP32 Demo</title>";
  #else
    #ifdef ESP8266_01
      msg += "<title>ESP8266-01 Demo</title>";
    #else
      msg += "<title>ESP8266-12 Demo</title>";
    #endif
  #endif
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  #ifdef ESP32
    msg += "<center><font size = \"3\"><b>ESP32 Demo Web Server</b></font><br>";
    msg += "This is an ESP32 demo webserver that has been running for ";
  #else
    msg += "<center><font size = \"3\"><b>ESP8266 Demo Web Server</b></font><br>";
    #ifdef ESP8266_01
      msg += "This is an ESP8266-01 demo webserver that has been running for ";
    #else
      msg += "This is an ESP8266-12 demo webserver that has been running for ";
    #endif
  #endif
  msg += millis()/1000;
  msg += " seconds.";
  msg += "<br>You can control the built-in LED with this webpage.";
  #ifdef __DateTime_RobotsForFun__
  msg += "<br>" + formattedTime();
  #endif
  msg += "</center>";
  msg += "<hr><br>";

  msg += "Current led status is ";
  switch(ledStatus) {
    case LEDOFF: 
      msg += "<b>OFF</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=ON\"><button style=\"font-size:16pt\">ON</button></a>";
      break;
    case LEDON: 
      msg += "<b>ON</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=BLINK\"><button style=\"font-size:16pt\">BLINK</button></a>";
      break;
    default: 
      msg += "<b>BLINK</b>"; 
      msg += "<br>Turn led&nbsp;";
      msg += "<a href=\"/\?LED=OFF\"><button style=\"font-size:16pt\">OFF</button></a>";
      break;
  }

  // another way to create the button
  msg += "<br><br>Another way to do it:";
  msg += "<form style=\"font-size:12pt\">";
  msg += "Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"OFF\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn off led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"ON\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to turn on led";
  msg += "<br>Click ";
  msg += "<input type=\"submit\" name=\"LED\" value=\"BLINK\" style=\"font-size:16pt\" >";
  msg += "&nbsp;to blink led";
  msg += "</form>";

  // wifi setup and info links
  msg += "<br><hr>IP address: ";
  if (wifiIsConnected) {
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
  } else {
    msg += "<a href=\"http://1.2.3.4\">1.2.3.4</a>";
    msg += "<br>Wifi setup and info: ";
    msg += "<a href=\"http://1.2.3.4/setup\">1.2.3.4/setup</a>";
  }
  if (internetIsConnected) { // Connect to DNS server to get IP address
    msg += "<br>Go to <a href=\"http://robotsforfun.com\">RobotsForFun.com</a> for more interesting projects.<br>";
    msg += "<center><a href=\"http://robotsforfun.com\"><img width=\"100\" src=\"http://robotsforfun.com/images/Robo.png\" alt=\"Robo the robot\"></a></center>";
  }
  msg += "</body>";
  msg += "</html>";
  // end of webpage file
  
  // send webpage to client browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client

}
#endif


// #define SETUPHTML  // define this to use your own setupHTML() function
#ifndef SETUPHTML
void setupHTML() {
  // get url arguments and process the command here
  // do it here to see the changes reflect in the webpage being sent back to the client
  // the code here should not have long delay times otherwise the webpage will be very slow to load
  if (server.hasArg("AP") && String(server.arg("AP")) == "OFF") {
    APisOn = false;
    turnOffAP();
  }
  if (server.hasArg("LED") && String(server.arg("LED")) == "OFF") {
    digitalWrite(statusLed, LEDOFF);  // turn off LED
    ledStatus = LEDOFF;
    Serial.println("turn off LED");
    EEPROM.put(EEPROM_ADDRESS_ledStatus, ledStatus);  // update ledStatus
    EEPROM.commit();
  } else if (server.hasArg("LED") && String(server.arg("LED")) == "BLINK") {
    ledStatus = LEDBLINK;                // blink LED
    Serial.println("blink LED");
    EEPROM.put(EEPROM_ADDRESS_ledStatus, ledStatus);  // update ledStatus
    EEPROM.commit();
  }

  if (server.hasArg("SYSTEMCOUNT")) {
    systemStartCount = 0;
    EEPROM.put(EEPROM_ADDRESS_systemStartCount, systemStartCount);  // update systemStartCount
    EEPROM.commit();
  }

  String msg = "";
  String m = "";
  msg += "<!DOCTYPE html>";
  msg += "<head>";
  msg += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">"; // keep fonts consistent between devices
  
  msg += "<title>WiFi Setup</title>";
  msg += "  <style>";
  msg += "    body { background-color: #ffffff; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }";
  msg += "  </style>";
  msg += "</head>";
  
  msg += "<body>";
  msg += "<h2>WiFi Setup and Info</h2>";

  if (wifiIsConnected) {
    msg += "Wifi ssid: ";
    msg += _ssid;
    msg += "<br>Password: ";
    msg += _password;
    // }
    msg += "<br>IP address: ";
    // m += WiFi.localIP()[0];
    // m += ".";
    // m += WiFi.localIP()[1];
    // m += ".";
    // m += WiFi.localIP()[2];
    // m += ".";
    // m += WiFi.localIP()[3];
    // msg += "<a href=\"http://" + m + "\">" + m + "</a>";
    msg += "<a href=\"/\">" + WiFi.localIP().toString() + "</a>";
    msg += "<br>Wifi setup: ";
    // msg += "<a href=\"http://" + WiFi.localIP().toString() + "/setup\">" + WiFi.localIP().toString() + "/setup</a>";
    msg += "<a href=\"/setup\">" + WiFi.localIP().toString() + "/setup</a>";
    msg += "<br>Port: ";
	msg += SERVERPORT;
    msg += "<br>Gateway: ";
    msg += WiFi.gatewayIP().toString();
	msg += "<br>AP ";
	if (APisOn) {
		msg += "ssid: ";
		msg += _APssid;
		msg += "<br>AP password: ";
		msg += _APpassword;
	} else {
		msg += "off";
	}
    msg += "<br>Hostname: ";
    #ifdef ESP32
      msg += WiFi.getHostname();
    #else
      msg += WiFi.hostname(); 
    #endif
    // print the received signal strength:
    long rssi = WiFi.RSSI();  // in dBm
    msg += "<br>Signal strength: RSSI ";
    msg += rssi;
    msg += " dBm ";
    msg += signalStrength();
    msg += "%";
    #ifndef ESP32
      msg += "<br>Vcc voltage: ";
      int p = (ESP.getVcc() + 5) / 10;
      msg += String(p/100.0);
      msg += "V";
    #endif
    msg += "<br>Mac address: ";
    msg += WiFi.macAddress();
    msg += "<br>Started on: ";
	#ifdef __DateTime_RobotsForFun__
    if (systemStartTime != 0) {
	  // time_t t = time(nullptr)
	  // struct tm *st = localtime(&t);
	  // msg += asctime(st);
      msg += formattedTime(systemStartTime);
    } else {
      msg += "not initialized";
    }
	#else
      msg += "not initialized";
	#endif	
    msg += "<br>Start count: ";
    msg += String(systemStartCount);
    // clear count
    msg += "&nbsp;<a href=\"/setup\?SYSTEMCOUNT=0\"><button style=\"font-size:14pt\">Reset Count</button></a>";

  } else {
    msg += "Wifi is not connected. Failed to connect to";
    msg += "<br>&nbsp;&nbsp;&nbsp;ssid: ";
    msg += _ssid;
    msg += "<br>&nbsp;&nbsp;&nbsp;password: ";
    msg += _password;
    msg += "<br>Main webpage: ";
    m += "1.2.3.4";
    msg += "<a href=\"http://" + m + "\">" + m + "</a>";

	msg += "<br>AP ";
	if (APisOn) {
		msg += "ssid: ";
		msg += _APssid;
		msg += "<br>AP password: ";
		msg += _APpassword;
	} else {
		msg += "off";
	}
	
    long rssi = WiFi.RSSI();  // in dBm
    msg += "<br>Signal strength: RSSI ";
    msg += rssi;
    msg += " dBm ";
    msg += signalStrength();
    msg += "%";
    #ifndef ESP32
      msg += "<br>Vcc voltage: ";
      int p = (ESP.getVcc() + 5) / 10;
      msg += String(p/100.0);
      msg += "V";
    #endif
    // msg += "<br>Start count: ";
    // msg += String(systemStartCount);
    // clear count
    // msg += "&nbsp;<a href=\"/setup\?SYSTEMCOUNT=0\"><button style=\"font-size:14pt\">Reset Count</button></a>";
  }
  msg += "<br><hr>";  
  int n = WiFi.scanNetworks();
  if (n > 0) {
	msg += "Available WiFi networks:<br>";
    for (int i = 0; i < n; i++) {
      #ifdef ESP32
        msg += "&nbsp;&nbsp;" + WiFi.SSID(i) + ((WiFi.encryptionType(i) == WIFI_AUTH_OPEN) ? " " : " *") + " (" + WiFi.RSSI(i) + ")<br>";
      #else
        msg += "&nbsp;&nbsp;" + WiFi.SSID(i) + ((WiFi.encryptionType(i) == ENC_TYPE_NONE) ? " " : " *") + " (" + WiFi.RSSI(i) + ")<br>";
      #endif
    }
  } else {
    msg += "No WiFi netwoks found<br>";
  }
  
  msg += "<hr>";  
  msg += "Enter info below to connect to ";
  if (wifiIsConnected) msg += "another ";
  msg += "WiFi:<br>";
  msg += "<form style=\"font-size:12pt\" >";
  msg += "  ssid:<br>";
  msg += "<input type=\"text\" name=\"SSID\" maxlength=\"30\" style=\"width:300px;font-size:14pt\" >";
  msg += "  <br>password:<br>";
  msg += "<input type=\"text\" name=\"PASSWORD\" maxlength=\"30\" style=\"width:300px;font-size:14pt\" >";
  msg += "  <br><br>";
  msg += "<input type=\"submit\" value=\"Submit\" style=\"font-size:14pt\">"; 
  // msg += "&nbsp;&nbsp;&nbsp;";
  // msg += "<a href=\"/setup\"><button style=\"font-size:14pt\">Clear</button></a>";
  msg += "</form>"; 
  msg += "<hr>";
  msg += "<a href=\"/setup?Reboot=ON\"><button style=\"font-size:14pt\" >Reboot</button></a>";
  if (APisOn) {
    msg += "&nbsp;&nbsp;<a href=\"/setup?AP=OFF\"><button style=\"font-size:14pt\" >Turn off AP</button></a>";
  }
  if (ledStatus != LEDOFF) {
    msg += "&nbsp;&nbsp;<a href=\"/setup?LED=OFF\"><button style=\"font-size:14pt\" >Turn off led</button></a>";
  } else {
    msg += "&nbsp;&nbsp;<a href=\"/setup?LED=BLINK\"><button style=\"font-size:14pt\" >Turn on led</button></a>";
  }

  msg += "<br>";
  #ifdef ESP32
	msg += "<a href=\"/updateCalled\"><button style=\"font-size:14pt\" >OTA Update Firmware</button></a>";
  #endif
  #ifdef ESP8266
	msg += "<a href=\"/update\"><button style=\"font-size:14pt\" >OTA Update Firmware</button></a>";
  #endif
  msg += "</body>";
  msg += "</html>";

  // send webpage to browser
  server.send(200, "text/html", msg);

  // get url arguments and process the command here
  // changes here will not be reflected in the webpage being sent (because it has already been sent)
  // therefore the code here can have long delay times since the webpage has already been sent back to the client
  if (server.hasArg("SSID") && server.arg("SSID").length() > 0) {
    server.arg("SSID").toCharArray(_ssid, SSID_SIZE);
    if (server.hasArg("PASSWORD")) {
      server.arg("PASSWORD").toCharArray(_password, PASSWORD_SIZE);
      saveWifiCredentials();
      ESP.restart();   
    }
  }
  if (server.hasArg("Reboot") && String(server.arg("Reboot")) == "ON") {
    Serial.println("Going to reboot");
    ESP.restart();   
  }

}
#endif


// #define NOTFOUNDHTML  // define this to use your own notfoundHTML() function
#ifndef NOTFOUNDHTML
void notfoundHTML() {
  setupHTML();
  // String msg = "File Not Found Error\n";
  // msg += " URI: ";
  // msg += server.uri();
  // msg += "\n Method: ";
  // msg += (server.method() == HTTP_GET)?"GET":"POST";
  // msg += "\n Arguments: ";
  // msg += server.args();
  // msg += "\n";
  // for (uint8_t i=0; i<server.args(); i++){
    // msg += " NAME:"+server.argName(i) + "\n VALUE:" + server.arg(i) + "\n";
  // }
  // server.send(404, "text/plain", msg);
}
#endif


//////////////////////////////////////////////////////////////
//// udp stuff

void sendPacket(const char packet[], int size, IPAddress ip, unsigned int port) {
  //IPAddress broadcastIP;
  //broadcastIP = ~WiFi.subnetMask() | WiFi.gatewayIP();
  //subnetMask: 255.255.255.0
  //gatewayIP: 192.168.1.1
  //broadcastIP: 192.168.1.255
  Serial.print("Sending \"");
  for (int i=0; i<size; i++) Serial.print(packet[i]);
  Serial.print("\" to ");
  Serial.print(ip);
  Serial.print(":");
  Serial.print(port);
  Serial.println("");
  udp.beginPacket(ip, port);
  // udp.write(packet, size);
  udp.printf(packet);
  udp.endPacket();
}

void sendPacket(const char packet[], int size) {
  sendPacket(packet, size, udp.remoteIP(), udp.remotePort());
}

void ackPacket(const char packet[], int size) {
  //IPAddress broadcastIP;
  //broadcastIP = ~WiFi.subnetMask() | WiFi.gatewayIP();
  //subnetMask: 255.255.255.0
  //gatewayIP: 192.168.1.1
  //broadcastIP: 192.168.1.255
  Serial.print("Ack ");
  sendPacket(packet, size, udp.remoteIP(), udp.remotePort());
}

// returns the number of bytes received
int receivePacket(char packet[]) {
  int packetSize = udp.parsePacket();
  if (packetSize) {
    // read the packet
    int len = udp.read(packet, packetSize);
    if (len > 0) {
      packet[len] = 0;  // insert null
    }
    Serial.print("Received \"");
    for (int i=0; i<len; i++) Serial.print(packet[i]);
    Serial.print("\" from ");
    Serial.print(udp.remoteIP());
    Serial.print(":");
    Serial.print(udp.remotePort());
    Serial.println();
    return packetSize;
  } else {
    return 0;
  }
}
#endif
