/*
 * This is the Music Player Controller and Hardware Serial Command Receiver
 * It waits for a command over the RX (D0) line
 * When it receives a command, it will execute the command which
 * controls the Music Player.
 *
 * The commands and format are:
 *  p# = play song at index #, e.g. p2 to play song at index 2
 *  s  = stop
 *  v# = volume adjust to #,   e.g. v0 to set volume to 0 (loudest)
 *  g1 or g0 = turn on/off the green led
 *  r1 or r0 = turn on/off the red led
 * All commands are terminated with '\0'
 *  
 * Copy the Music_Shield_V2_0 to the Arduino libraries folder
 * Compiles using Arduino v1.0.5

 * This Music_Shield_V2_0 library is different from the original downloaded library
 * with added functions of my own

 * Can use any microSD card
 * Must use MP3 files
 */


//////////////////////////////////////////////////////////////
//// Music player stuff
#include <SD.h>
#include <SPI.h>
#include <arduino.h>
#include <MusicPlayer.h>


//////////////////////////////////////////////////////////////
//// status LED stuff
#define greenLED 8
#define redLED 9
int activityCount = 0;
void flashActivityLed(int count, int wait=150) {
  // flash activity LED
  if (!digitalRead(greenLED)) {
    for (int i=0; i<count; i++ ) {
      digitalWrite(greenLED, HIGH);//GREEN_LED_ON();
      delay(wait);
      digitalWrite(greenLED, LOW);//GREEN_LED_OFF();
      delay(wait);
    }
  }
}

//////////////////////////////////////////////////////////////
//// Music Player stuff
#define DOORSONGS  11
//#define TOTALSONGS 127
int TOTALSONGS;
boolean shuffle = false;  // flag for shuffle play
boolean continuous = false;  // flag for continuous play
int currentSong;              // for continuous play

//////////////////////////////////////////////////////////////
//// interface with ESP8266 MusicPlayer_Receiver stuff
// connect RX pin (D0) on the Arduino to the TX pin (D1) on the ESP8266
char command[4];
boolean connectionEstablished = false;
boolean firstTime = true;


void executeCommand(char command[]) {
  char cmd;
  int number;
    cmd = command[0];
    command[0] = ' ';
    number = atoi(command);
    
    
    if (cmd == 'p') {       // play song
      connectionEstablished = true;
      shuffle = false;
      continuous = false;
      player.stopSong();
      player.playSong(number);
    }
    
    else if (cmd == 'g') {  // set green led
      connectionEstablished = true;
      Serial.print("Set green led ");
      if (number == 1) {
        GREEN_LED_ON();
        Serial.println("on");
      } else if (number == 0) {
        GREEN_LED_OFF();
        Serial.println("off");
      } else {
        Serial.println(number);
      }
    }
    
    else if (cmd == 'r') {  // set red led
      connectionEstablished = true;
      Serial.print("Set red led ");
      if (number == 1) {
        RED_LED_ON();
        Serial.println("on");
      } else if (number == 0) {
        RED_LED_OFF();
        Serial.println("off");
      } else {
        Serial.println(number);
      } 
    }
    
    else if (cmd == 'v') {  // set volume
      connectionEstablished = true;
      Serial.print("Set volume to ");
      Serial.println(number);
      player.setVolume(number);  // 0 is loudest
    }
  
    else if (cmd == 's') {  // stop playing
      connectionEstablished = true;
      continuous = false;
      shuffle = false;
      Serial.println("Stop playing ");
      player.stopSong();
    }
  
    else if (cmd == 'c') {  // continuous
      currentSong = number;
      if (currentSong >= TOTALSONGS) currentSong = DOORSONGS;
      continuous = true;
      //Serial.println("Connection established");
    }
    
    else if (cmd == 'f') {  // shuffle play
      randomSeed(number);
      shuffle = true;

    } // end if  
}

// this routine is used to read in the serial data
// it is automatically called whenever there's data in the serial stream
void serialEvent() {
  static int count = 0;
  //Serial.print("in serialEvent - ");
  command[count] = Serial.read();
  //digitalWrite(greenLED, HIGH);
  Serial.print(command[count]);
  if (command[count] == '\0') {
    Serial.print(" ");
    count = 0;
    executeCommand(command);
    //digitalWrite(greenLED, LOW);
  } else {
    count++;
  }
}


/*
#ifdef __checkSensor__
//////////////////////////////////////////////////////////////
//// Sensors stuff
#define beamSensor A4
#define motionSensor A5
boolean beamStatus, motionStatus;
unsigned long alarmOn = 0;

void checkSensors() { // check the two front door sensors
  
  // alarm has been triggered; delay before resetting
  if (alarmOn > 0) {
    if (millis() < alarmOn) {
      return;  // this will skip checking the sensors
    } else {
      alarmOn = 0;
      activityCount = 0;
    }
  }
  
  beamStatus = digitalRead(beamSensor);
  motionStatus = digitalRead(motionSensor);

  // check for coming in
  if((beamStatus == LOW) && (motionStatus == HIGH)) {
    RED_LED_ON();
    for(int i=0; i<70; i++) {
      // wait for motion sensor trigger
      flashActivityLed(1,50);
      if(digitalRead(motionSensor) == LOW) {
        // motion sensor triggered
        Serial.println();
        Serial.println("Motion sensor triggered");
        alarmOn = millis()+5000;  // time to reset

        //execute("s  ");   // stop song first
        //if ((hour() > 8) && (hour() < 17)) { // day
          //execute("v0 ");  // set volume; 0=max
        //} else {
          //execute("v30");  // set volume softer
        //}

        //execute("p1 ");
        if (player.idle()) {
          player.playSong(1);
          delay(12000);
          //execute("p1");  // play warning again after 12 seconds
          //delay(12000);
          
          //execute("s  ");   // stop song first
          if (currentSong == DOORSONGS) currentSong = 2;
          currentSong++;
          //String song = "p" + String(currentSong); // convert int to string
          //execute(song);
          player.playSong(currentSong);
        }
        break;
      }      
      delay(100);
    } // end for
    //execute("r0 ");
    RED_LED_OFF();

  }
    
  // check for going out
  else if((beamStatus == HIGH) && (motionStatus == LOW)) {
    //execute("r1 ");
    RED_LED_ON();
    for(int i=0; i<80; i++) {
      // wait for beam sensor trigger      
      flashActivityLed(1,50);
      if(digitalRead(beamSensor) == LOW) {
        // beam sensor triggered
        Serial.println();
        Serial.println("Beam sensor triggered");
        alarmOn = millis()+5000;  // time to reset
        //execute("s  ");   // stop song first

        //if ((hour() > 8) && (hour() < 17)) { // day
          //execute("v10");  // set volume; 0=max
        //} else {
          //execute("v30");  // set volume softer
        //}

        //execute("p0 ");
        if (player.idle()) {
          player.playSong(0);
        }
        break;
      }
      delay(100);
    } // end for
    //execute("r0 ");
    RED_LED_OFF();

  } // end if check for sensors activities  
}
#endif
*/

void setup() {
  Serial.begin(115200);
  pinMode(greenLED, OUTPUT);
  pinMode(redLED, OUTPUT);

/*
#ifdef __checkSensor__
  pinMode(beamSensor, INPUT_PULLUP);
  pinMode(motionSensor, INPUT_PULLUP);
#endif
*/

  //// Music player stuff
  player.begin();                  // will initialize the hardware and set default mode to be normal.
  player.setVolume(0);            // 0 is loudest to about 50 softest
  player.setPlayMode(PM_PLAY_ONE); // set mode to play song once
  //player.keyDisable();             // disable key checking
  TOTALSONGS = player.totalNumberOfSongs();
  Serial.print("Total number of songs on the SD card is ");
  Serial.println(TOTALSONGS);
  player.playSong(0);              // play song 0
  //while(!player.idle());
  //player.playSong("STARWARS.MP3");              // play song 0

  Serial.println("Waiting for hardware serial connection on RX...");

}

void loop() {
  //checkSensors(); // check the two front door sensors

  if (!connectionEstablished) {
    digitalWrite(greenLED, !digitalRead(greenLED));
    delay(200);
    digitalWrite(redLED, !digitalRead(redLED));
    delay(200);
  } else if (firstTime) {
    firstTime = false;
    digitalWrite(greenLED, LOW);
    digitalWrite(redLED, LOW);
    Serial.println("Ready");
  }

  if (continuous && player.idle()) {
    player.playSong(currentSong);  // soft music songs on the SD card
    currentSong++;
    if (currentSong >= TOTALSONGS) currentSong = DOORSONGS;
  }

  if (shuffle && player.idle()) {
    player.playSong(random(DOORSONGS,TOTALSONGS));  // soft music songs on the SD card
  }

/*  
  // show activity light
  activityCount++;
  if (activityCount >= 500) {
    activityCount = 0;
    flashActivityLed(2);
  }
*/
}

