// Barebone example for WiFi NTP clock time
//
// Reference for struct tm as defined in <time.h>
// http://www.cplusplus.com/reference/ctime/
// http://www.cplusplus.com/reference/ctime/tm/
// only works with ESP8266 v2.4.0 or higher
// select from the menu Tools | Boards | Board Manager
// then install the ESP8266 v2.4.0 or higher
//
// Reference for configTime
// https://github.com/espressif/arduino-esp32/blob/f41beb92bf4434f52209bd15655c1b2f2447447f/cores/esp32/esp32-hal-time.c
// https://github.com/esp8266/Arduino/blob/9913e5210779d2f3c4197760d6813270dbba6232/cores/esp8266/time.c
// https://github.com/esp8266/Arduino/blob/61cd8d83859524db0066a647de3de3f6a0039bb2/libraries/esp8266/examples/NTP-TZ-DST/NTP-TZ-DST.ino
//
// Note that if the clock is not automatically set initially but just manually set
// then the localtime() and gmtime() are both the same, and the dst flag has no effect.
// However, if the clock is first automatically set then the localtime() is
// for the given timezone whereas the gmtime() is for UTC, and the dst flag will affect the time.
//
// Copyright (c) 2020 Enoch Hwang

#include <ESP8266WiFi.h>
#include <time.h>
#include <coredecls.h>     // only need this if using settimeofday_cb() callback

char ssid[] = "your ssid";           // change this to your ssid
char password[] = "your password";   // change this to your password
//bool dst = true;   // true=daylight savings time; false=standard time
String weekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};

void printTime(struct tm *ptm) {
  Serial.print(weekdays[ptm->tm_wday]);
  Serial.print(" ");
  Serial.print(ptm->tm_year+1900);
  Serial.print("/");
  Serial.print(ptm->tm_mon+1);
  Serial.print("/");
  Serial.print(ptm->tm_mday);
  Serial.print(" ");
  Serial.print(ptm->tm_hour);
  Serial.print(":");
  Serial.print(ptm->tm_min);
  Serial.print(":");
  Serial.print(ptm->tm_sec);
//  Serial.print(" ");
//  Serial.print(asctime(ptm));  
}

void printTime() {
  // get current raw time
  time_t now = time(nullptr);

  // convert current time to local time
  struct tm *ptm;
  ptm = localtime(&now); // convert time_t to tm structure as local time
  Serial.print("Local time: ");
  printTime(ptm);

  // convert current time to UTC time
  ptm = gmtime(&now); // convert time_t to tm structure as UTC time
  Serial.print("  UTC time: ");
  printTime(ptm);

  Serial.print("  (");
  Serial.print(now);
  Serial.println(")");
}

// time is valid if > Thu Jan  1 08:00:03 1970
bool isTimeValid() { 
  // when the clock is not set time(nullptr) returns the value 28803
  // which is for Thu Jan  1 08:00:03 1970
  // so a valid time is greater than 28803
  static time_t SECS_START_POINT = 30000;
  return time(nullptr) > SECS_START_POINT;
}

// The clock is automatically set after there's a network connection
// localtime() and gmtime() are different when setting the system clock automatically using configTime()
void setClockAuto() {
  // implement NTP update of timekeeping (with automatic hourly updates and daylight savings time adjustment)
  // https://github.com/espressif/arduino-esp32/blob/f41beb92bf4434f52209bd15655c1b2f2447447f/cores/esp32/esp32-hal-time.c
  // this call replaces the calls to
  //  configTime(0, 0, "0.pool.ntp.org");
  // and
  //  setenv("TZ", "PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", 1); 
  // See the following on how to set the timezone TZ
  // https://www.gnu.org/software/libc/manual/html_node/TZ-Variable.html
  // PST is +8 hours from UTC
  // Daylight savings starts on M3.2.0/2:00:00 = March 2nd Sunday at 2:00:00 am
  //  3=March, 2=2nd, 0=Sunday, 2=hour, 00=minute, 00=second
  // and ends on M11.1.0/2:00:00 = November 1st Sunday at 2:00:00 am
  //  11=November, 1=1st, 0=Sunday
  configTzTime("PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", "0.pool.ntp.org");
    
  // optional: register a callback (execute whenever an NTP update has occurred)
  // https://forum.arduino.cc/t/esp8266-null-sketch-finds-time-why-how/627573/9
  settimeofday_cb(time_is_set);

  // connect to wifi
  WiFi.begin(ssid, password);
  Serial.print("Connecting to ");
  Serial.print(ssid);
  while (WiFi.status() != WL_CONNECTED) {
    Serial.print(".");
    delay(150);
  }
  Serial.println("connected");
  Serial.print("IP address is ");
  Serial.println(WiFi.localIP());
  
  // wait for time to be valid
  while (!isTimeValid()) {
    Serial.print(".");
    delay(150);
  }
  Serial.print("Clock automatically set to ");
  time_t now = time(nullptr);
  Serial.println(ctime(&now));  
}

// localtime() and gmtime() are both the same when setting the system clock manually using settimeofday()
void setClockManual(const unsigned int yr, const unsigned int mo, const unsigned int da, const unsigned int hr, const unsigned int mi, const unsigned int se = 0) {
  struct tm st;    
  st.tm_year = yr - 1900; /* The number of years since 1900   */
  st.tm_mon = mo - 1;     /* month, range 0 to 11             */
  st.tm_mday = da;        /* day of the month, range 1 to 31  */
  st.tm_hour = hr;        /* hours, range 0 to 23             */
  st.tm_min = mi;         /* minutes, range 0 to 59           */
  st.tm_sec = se;         /* seconds,  range 0 to 59          */
  st.tm_isdst = -1;       /* daylight savings time -1=unknown. Changing this has no effect */
  time_t t = mktime(&st); /* convert tm structure to time_t   */

  // set the system clock
  struct timeval tv;
  tv.tv_sec = t;
  settimeofday(&tv, NULL);  // set the system clock

  Serial.print("Clock manually set to ");
  time_t now = time(nullptr);
  Serial.println(ctime(&now));  
}

// callback routine - arrive here whenever a successful NTP update has occurred
void time_is_set (void) {
  Serial.print("\n-------- NTP updated at ");
  time_t now = time(nullptr);
  Serial.println(ctime(&now));  
}

void setup() {
  Serial.begin(115200);
  Serial.println();
  setClockAuto();
  printTime();
  Serial.println();
  setClockManual(2000, 9, 3, 10, 35, 0);
  printTime();
}

void loop() {
  printTime();
  delay(1000);
}
