//////////////////////////////////////////////////////////
// NTP (Network Time Protocol) Date Time class
// Clock is automatically set using NTP
//

// This library only works with ESP8266 v2.4.0 or higher
// select from the Arduino IDE menu Tools | Boards | Board Manager
// then install the ESP8266 v2.4.0 or higher

// Time is stored either as time_t or struct tm
// time_t is referred to as machine time or unix time.
// It is generally implemented as the number of seconds elapsed since 
// 00:00 hours, Jan 1, 1970 UTC.
//
// struct tm is referred to as human readable time which is a structure 
// containing a calendar date and time broken down into its components
//   Member		Type	Meaning								Range
//   tm_sec		int		seconds								0-59
//   tm_min		int		minutes								0-59
//   tm_hour	int		hours									0-23
//   tm_mday	int		day of the month			1-31
//   tm_mon		int		months								0-11 0=January
//   tm_year	int		years since 1900	
//   tm_wday	int		days of the week 			0-6 0=Sunday
//   tm_yday	int		days since January 1	0-365
//   tm_isdst	int		Daylight Saving Time flag	1=DST, 0=Standard Time, -1=unknown
//
// References for time_t and struct tm as defined in <time.h>
// http://www.cplusplus.com/reference/ctime/
// https://www.tutorialspoint.com/c_standard_library/time_h.htm
//
// References for configTime()
// https://github.com/esp8266/Arduino/blob/9913e5210779d2f3c4197760d6813270dbba6232/cores/esp8266/time.c
// https://github.com/esp8266/Arduino/blob/61cd8d83859524db0066a647de3de3f6a0039bb2/libraries/esp8266/examples/NTP-TZ-DST/NTP-TZ-DST.ino
//
// Copyright (c) 2020 Enoch Hwang

#ifndef __DateTime_RobotsForFun__
#define __DateTime_RobotsForFun__
//#include <Arduino.h>

#ifdef ESP32 
  #include <WiFi.h>	        // for calling WiFi.status()
#else
  #include <ESP8266WiFi.h>	// for calling WiFi.status()
  #include <coredecls.h>     // only need this if using settimeofday_cb() callback
#endif
#include <time.h>

#define PST -8
#define MST -7
#define CST -6
#define EST -5
#define AST -4
#define BEIJING 8
#define GMT 0
#define UTC 0
#define HAWAII -11
#define HONGKONG 8
#define LONDON 0
#define LOSANGELES -8
#define MEXICOCITY -6
#define PARIS 1
#define TOKYO 12
#define TORONTO -5
#define VANCOUVER -8


//unsigned long syncInterval = 3600000;  // In ms  default is 1 hour
//unsigned long lastSync     = 0;        // In ms
//int timeZone = PST;        // Pacific Standard Time (USA)
//bool DST     = false;      // Daylight Savings Time

void printStruct(struct tm st) {
	Serial.print(st.tm_mon+1);
	Serial.print("/");
	Serial.print(st.tm_mday);
	Serial.print("/");
	Serial.print(st.tm_year+1900);
	Serial.print(" ");
	Serial.print(st.tm_hour);
	Serial.print(":");
	Serial.print(st.tm_min);
	Serial.print(":");
	Serial.println(st.tm_sec);
}

// returns a human readable time string from a struct tm
String formattedTime(struct tm st, bool shortFormat = false, bool hour12 = true) {
  time_t t = mktime(&st);
  if (t == -1) {
    Serial.println("Error in getFormattedTime");
    return "";
  }
  static String wd[] = {"Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"};
  String s = "";
  if (shortFormat) {
    // 2020-02-14 08:35:40
    s = s + String(st.tm_year+1900) + "-";  
    if (st.tm_mon+1 < 10) s = s + "0";
    s = s + String(st.tm_mon+1) + "-";
    if (st.tm_mday < 10) s = s + "0";
    s = s + String(st.tm_mday) + " ";
    if (st.tm_hour < 10) s = s + "0";
    s = s + String(st.tm_hour) + ":";
    if (st.tm_min < 10) s = s + "0";
    s = s + String(st.tm_min) + ":";
    if (st.tm_sec < 10) s = s + "0";
    s = s + String(st.tm_sec);
  } else { // default format    
    // 11:44:53 am Fri 2/28/2020 hh:mm:ss am ddd mm/dd/yyyy
    int h = st.tm_hour;
    String ampm = "";  
    if (hour12) {
      if (h >= 12) {
        ampm = "pm";
        if (h > 12) {
          h = h - 12; 
        }
      } else {
        ampm = "am";
      }
    }
    if (h < 10) s = s + "0";
    s = s + String(h) + ":";
    if (st.tm_min < 10) s = s + "0";
    s = s + String(st.tm_min) + ":";
    if (st.tm_sec < 10) s = s + "0";
    s = s + String(st.tm_sec) + " " + ampm + " ";

    s = s + wd[st.tm_wday].substring(0,3) + " ";
    //s = s + wd[st.tm_wday] + " ";
    s = s + String(st.tm_mon+1) + "/";
    s = s + String(st.tm_mday) + "/";
    s = s + String(st.tm_year+1900) + " ";
  }
  return s;  
}

// returns a human readable time string from a time_t machine time
String formattedTime(time_t t = 0, bool shortFormat = false, bool hour12 = true) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return formattedTime(st, shortFormat, hour12);
}

// returns the year given a struct tm
int year(struct tm st) {
  return st.tm_year + 1900;
}

// returns the year since 1900 given a time_t
// returns the current year since 1900 if t=0
int year(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_year + 1900;
}

// returns the month given a struct tm
// 0 = January
int month(struct tm st) {
  return st.tm_mon + 1;   // 0 = January
}

// returns the month given a time_t
// returns the current month if t=0
// 0 = January
int month(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  // struct tm st = *gmtime(&t);
  struct tm st = *localtime(&t);
  return st.tm_mon + 1;   // 0 = January
}

// returns the day given a struct tm
int day(struct tm st) {
  return st.tm_mday;
}

// returns the day of the month given a time_t
// returns the current day if t=0
int day(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_mday;
}

// returns the weekday given a struct tm
int weekday(struct tm st) {
  return st.tm_wday;    // 0 = Sunday
}

// returns the weekday given a time_t
// returns the current weekday if t=0
// 0=Sunday
int weekday(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_wday;    // Sunday = 0
}

// returns the hour (military time) given a struct tm
int hour(struct tm st) {
  return st.tm_hour;
}

// returns the hour (military time) given a time_t
// returns the current hour if t=0
int hour(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_hour;
}

// returns the hour (military time) given a struct tm
int hour24(struct tm st) {
  return st.tm_hour;
}

// returns the hour (military time) given a time_t
// returns the current hour if t=0
int hour24(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_hour;
}

// returns the hour given a struct tm
int hour12(struct tm st) {
  if(st.tm_hour == 0)
    return 12; // 12 midnight
  else if(st.tm_hour  > 12)
    return st.tm_hour - 12;
  else
    return st.tm_hour;
}

// returns the hour given a time_t
// returns the current hour if t=0
int hour12(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return hour12(st);
}

// return true if AM given a struct tm
bool isAM(struct tm st) { // returns true if PM
  return (st.tm_hour < 12); 
}

// return true if AM given a time_t
bool isAM(time_t t = 0) { // returns true if PM
  return (hour(t) < 12); 
}

// return true if PM given a struct tm
bool isPM(struct tm st) { // returns true if PM
  return (st.tm_hour >= 12); 
}

// return true if PM given a time_t
bool isPM(time_t t = 0) { // returns true if PM
  return (hour(t) >= 12); 
}

// returns the minute given a struct tm
int minute(struct tm st) {
  return st.tm_min;
}

// returns the minute given a time_t
// returns the current minute if t=0
int minute(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_min;
}

// returns the second given a struct tm
int second(struct tm st) {
  return st.tm_sec;
}

// returns the second given a time_t
// returns the current second if t=0
int second(time_t t = 0) {
  if (t == 0) {
    t = time(nullptr);  // get current date/time
  }
  struct tm st = *localtime(&t);
  return st.tm_sec;
}

// returns the current time time_t
time_t now() {
  return time(nullptr);
}

/*
// returns true if current time is in daylight savings time
// else returns false
// daylight savings time is from 
//   2nd Sunday in March at 2 am (March 8, 2020)
// to
//   1st Sunday in November at 1:59:59 am (November 1, 2020)
bool isDaylightSavingsTime() {
  struct tm st;
  time_t t,t3,t11;

  // find seconds for 2nd Sunday in March for current year at 2 am
  // m == 3 && d >= 8 && d <= 14 && w == 0
  st.tm_hour = 2;
  st.tm_min = 0;
  st.tm_sec = 0;
  st.tm_mday = 8;
  st.tm_mon = 2;  // 2 = March; 0 = January
  st.tm_year = year() - 1900;
  t3 = mktime(&st);
  int w = weekday(t3);
  if (w > 0) { // 0 = Sunday
    st.tm_mday = st.tm_mday + (7 - w); // next Sunday
  }
  t3 = mktime(&st);     // seconds for 2nd Sunday in March at 2 am
  //Serial.print("t3=");
  //Serial.println(getFormattedTime(t3));

  // find seconds for 1st Sunday in November for current year at 2 am
  // m == 11 && d < 8 && w == 0
  st.tm_hour = 2;
  st.tm_min = 0;
  st.tm_sec = 0;
  st.tm_mday = 1;
  st.tm_mon = 10;  // 10 = November; 0 = January
  st.tm_year = year() - 1900;
  
  t11 = mktime(&st);
  w = weekday(t11);
  if (w > 0) { // 0 = Sunday
    st.tm_mday = st.tm_mday + (7 - w); // next Sunday
  }
  t11 =  mktime(&st); // seconds for 1st Sunday in November at 2 am
  //Serial.print("t11=");
  //Serial.println(getFormattedTime(t11));

  // seconds for current time
  t = time(nullptr);
  //t = toMachineTime(2020,11,1,2,0,0); // for testing
  //t=t-1;
  //Serial.print("t=");
  //Serial.println(getFormattedTime(t));
  
  if (t >= t3 && t < t11) {
    //Serial.println("Daylight Savings Time");
    return true;
  } else {
    //Serial.println("Standard Time");
    return false;;
  }  
}
*/

/*
// set the DST flag from the isDaylightSavingsTime() call
// this function is called from syncClock() which must be called very often
void checkDST() {
  // check standard time or daylight savings time at 2 AM each day
  // if (hour() == 2) {	  
  if ((hour() == 2) && (minute() == 0) && (second() == 0)) {
    bool currentDST = DST;
    DST = isDaylightSavingsTime();
    if (DST != currentDST) {
      time_t currentTime = time(nullptr);
      Serial.print("Adjusting time change");
	  unsigned long timeout = millis() + 20000; // 20 seconds
	  configTime(timeZone*3600, DST*3600, "us.pool.ntp.org", "time.nist.gov");
      while ((abs(time(nullptr) - currentTime) < 3600) && (millis() < timeout)) {
        digitalWrite(statusLed, !digitalRead(statusLed));
        nonblockingDelay(300);
        Serial.print(".");
      }
      Serial.println(formattedTime());
    }
  }
}
*/

// convert from year, month, day, hour, minute, second to time_t
time_t toMachineTime(const unsigned int yr, const unsigned int mo, const unsigned int da, const unsigned int hr=0, const unsigned int mi=0, const unsigned int se=0) {
  struct tm st;		
  st.tm_year = yr - 1900; /* The number of years since 1900   */
  st.tm_mon = mo - 1;     /* month, range 0 to 11             */
  st.tm_mday = da;        /* day of the month, range 1 to 31  */
  st.tm_hour = hr;        /* hours, range 0 to 23             */
  st.tm_min = mi;         /* minutes, range 0 to 59           */
  st.tm_sec = se;         /* seconds,  range 0 to 59          */
  st.tm_isdst = -1;      /* daylight savings time -1=unknown */
  time_t t = mktime(&st);
	if ((t == -1) || (yr < 1970) || (mo > 11) || (da == 0) || (da > 31) || (hr > 23) || (mi > 59) || (se > 59)) {
    Serial.println("mktime error");
	return 0;
  }
  return t;
}

// convert from struct tm to time_t
time_t toMachineTime(struct tm st) {
  time_t t = mktime(&st);
  if (t == -1) {
    Serial.println("mktime error");
	return 0;
  }
  return(t);
}

// convert from time_t to struct tm
// same as gmtime()
struct tm toHumanTime(const time_t t) {
  //return *gmtime(&t);
  return *localtime(&t);
}

/*
// set how often to sync the clock in ms
bool setSyncInterval(int si = 3600000) {	
  if (si >= 1000) {
	syncInterval = si;
	return true;
  } else {
	return false;
  }
}

bool setTimeZone(int tz = PST) {
  if (tz >= -11 || tz <= 13) {
    timeZone = tz;
    return true;
  } else {
	return false;
  }
}
*/

// time is valid if > Thu Jan  1 08:00:03 1970
bool isTimeValid() { 
  // when the clock is not set time(nullptr) returns the value 28803
  // which is for Thu Jan  1 08:00:03 1970
  // so a valid time is greater than 28803
  static time_t SECS_START_POINT = 30000;
  return time(nullptr) > SECS_START_POINT;
}

bool waitForTimeValid() {
  // check if there's internet connection
  IPAddress ipaddress;
  unsigned long startTime = millis();
  unsigned long nextLedActivity = millis();  
  if (WiFi.hostByName("www.google.com", ipaddress) == 1) { // Connect to DNS server to get IP address
    Serial.print("Waiting for clock to sync");
    while(!isTimeValid() && (millis() - startTime < 30000)) {
      //server.handleClient();  // handle web client request	// server not defined at this point
      yield();
      if (millis() - nextLedActivity >= 300) {
        nextLedActivity = millis();
        digitalWrite(statusLed, !digitalRead(statusLed));
        Serial.print(".");
      }
    }
    Serial.println();
    if (isTimeValid()) {
      return true;
    }
  }
  return false;
}



// callback function used in setClock() 
// This function is automatically called whenever a successful NTP update has occurred
void time_is_set () {
  Serial.print("\nNTP updated on ");
  time_t now = time(nullptr);
  Serial.println(ctime(&now));  
}

// The clock is automatically set after there's a network connection
// localtime() and gmtime() are different when setting the system clock automatically using configTime()
// cb_function is the name of the function for the callback
// Usage: setClock(PST, yourFunction);
void setClock(int tz, void (*cb_function)(void)) {
  // implement NTP update of timekeeping (with automatic hourly updates and daylight savings time adjustment)
  // https://github.com/espressif/arduino-esp32/blob/f41beb92bf4434f52209bd15655c1b2f2447447f/cores/esp32/esp32-hal-time.c
  // this call replaces the calls to
  //  configTime(0, 0, "0.pool.ntp.org");
  // and
  //  setenv("TZ", "PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", 1); 
  // See the following on how to set the timezone TZ
  // https://www.gnu.org/software/libc/manual/html_node/TZ-Variable.html
  // PST is +8 hours from UTC
  // Daylight savings starts on M3.2.0/2:00:00 = March 2nd Sunday at 2:00:00 am
  //  3=March, 2=2nd, 0=Sunday, 2=hour, 00=minute, 00=second
  // and ends on M11.1.0/2:00:00 = November 1st Sunday at 2:00:00 am
  //  11=November, 1=1st, 0=Sunday
  switch(tz) {
	case PST:
      configTzTime("PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", "0.pool.ntp.org");
	  break;
	case UTC:
      configTzTime("UTC0UTC", "0.pool.ntp.org");
	  break;
	default:
      configTzTime("PST+8PDT,M3.2.0/2:00:00,M11.1.0/2:00:00", "0.pool.ntp.org");
	  break;
  }
#ifndef ESP32
  // optional: register a callback (execute whenever an NTP update has occurred)
  // https://forum.arduino.cc/t/esp8266-null-sketch-finds-time-why-how/627573/9
  // settimeofday_cb(time_is_set);
  settimeofday_cb(cb_function);
#endif
}

// Use default callback function time_is_set()
// Usage: setClock(PST);
void setClock(int tz=PST) {
  setClock(tz,  time_is_set);

}

// manually set the clock with given date/time info
// returns true if successful
// Usage: setClock(2021, 5, 18, 10, 9);
bool setClock(const unsigned int yr, const unsigned int mo, const unsigned int da, const unsigned int hr, const unsigned int mi, const unsigned int se = 0) {
  struct timeval tv;
  tv.tv_sec = toMachineTime(yr, mo, da, hr, mi, se);
  if (tv.tv_sec == 0) return false;
//  settimeofday(&tv, NULL);	// always interprets the time in tv as UTC time
/*   if (isTimeValid()) {
	Serial.println("here 1");
    // however, if the clock is already set, 
    // then the time is interpreted as UTC+timeZone
    // so need to subtract the timeZone*3600 seconds
    tv.tv_sec = tv.tv_sec - timeZone*3600;
  } else {
	Serial.println("here 2");
    // and if the clock is not yet set, 
    // then the time is interpreted as just UTC
    // so need to add the timeZone*3600 seconds
    //tv.tv_sec = tv.tv_sec + timeZone*3600;
    tv.tv_sec = tv.tv_sec;
  }
 */
  settimeofday(&tv, NULL);  // set the system clock
  Serial.print("Clock manually set to ");
  Serial.print(formattedTime());
  Serial.print(" ");
//  DST = isDaylightSavingsTime();
  return true;
}

/*
// automatically set the clock with given time zone
// returns true if successful
bool setClock(int tz = PST) {
  if (WiFi.status() == WL_CONNECTED) {
    Serial.print("Setting clock");
	timeZone = tz;
    unsigned long timeout = millis() + 20000; // 20 seconds
    configTime(timeZone*3600, DST*3600, "us.pool.ntp.org", "time.nist.gov");
    while (!isTimeValid() && (millis() < timeout)) {
      digitalWrite(statusLed, !digitalRead(statusLed));
      nonblockingDelay(300);
      Serial.print(".");
    }
    Serial.println();
	if (isTimeValid()) {
      time_t currentTime = time(nullptr);
	  lastSync = millis();
	  Serial.print("Clock automatically set to ");
	  Serial.println(formattedTime());
      // at this point the clock is set to standard time because DST is initialized to false
	  DST = isDaylightSavingsTime();
      // if DST is true then need to wait for clock to adjust by adding 1 hour
      if (DST) {
        Serial.print("Adjusting for daylight savings time");
        timeout = millis() + 20000; // 20 seconds
        configTime(timeZone*3600, DST*3600, "us.pool.ntp.org", "time.nist.gov");
        while ((time(nullptr) < currentTime+3600) && (millis() < timeout)) { //
          digitalWrite(statusLed, !digitalRead(statusLed));
          nonblockingDelay(300);
          Serial.print(".");
        }
        if (timeout <= 0) {
          Serial.println();
          Serial.println("Clock not adjusted to daylight savings time");
          return false;
        }
        Serial.println(formattedTime());
      }
	  return true;
	} else {
	  Serial.println();
	  Serial.println("Clock not set");
	  return false;
	}
  } else {
    // Serial.println("No internet");
    return false;
  }
}
*/

/*
// need to call this regularly in the main loop
void syncClock() {
  checkDST();
  if ((millis() > lastSync + syncInterval) || (lastSync == 0)) { 
    // checkDST();
    configTime(timeZone*3600, DST*3600, "us.pool.ntp.org", "time.nist.gov");
    lastSync = millis();
    Serial.print("Clock syncronized on ");
	Serial.println(formattedTime());
  }
}
*/
#endif
