#include <ESP8266WiFi.h>
#include <WiFiUdp.h>
#define SECS_PER_HOUR 3600UL
#define SEVENZYYEARS 2208988800UL
#define NTP_PACKET_SIZE 48
#define NTP_DEFAULT_LOCAL_PORT 8888
#define PST -8
const char *ssid     = "your ssid";
const char *password = "your password";
const char *poolServerName = "us.pool.ntp.org";
byte packetBuffer[NTP_PACKET_SIZE];  //buffer to hold incoming & outgoing packets
unsigned long syncInterval = 10000;  // In ms
unsigned long lastSync     = 0;      // In ms
unsigned long currentEpoc  = 0;      // In s
int timeZone = PST;        // Pacific Standard Time (USA)
bool DST     = false;      // Daylight Savings Time

WiFiUDP udp;

unsigned long getEpochTime() {
  return currentEpoc +                     // Epoc returned by the NTP server
         ((millis() - lastSync) / 1000);   // Time since last update
}

void printFormattedTime() {
  time_t t = getEpochTime();
  struct tm st;
  st = *gmtime(&t);
  // Serial.println(asctime(&st));
  Serial.print(t);
  Serial.print(" ");
  String week[] = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
  Serial.print(week[st.tm_wday]);
  Serial.print(" ");
  Serial.print(st.tm_mon+1);
  Serial.print("/");
  Serial.print(st.tm_mday);
  Serial.print("/");
  Serial.print(st.tm_year+1900);
  Serial.print(" ");
  if (st.tm_hour < 10) Serial.print("0");
  Serial.print(st.tm_hour);
  Serial.print(":");
  if (st.tm_min < 10) Serial.print("0");
  Serial.print(st.tm_min);
  Serial.print(":");
  if (st.tm_sec < 10) Serial.print("0");
  Serial.println(st.tm_sec);
}

void sendNTPPacket() {
  IPAddress ntpServerIP; // NTP server's ip address
  WiFi.hostByName(poolServerName, ntpServerIP);

  // set all bytes in the buffer to 0
  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  // Initialize values needed to form NTP request
  // (see URL above for details on the packets)
  packetBuffer[0] = 0b11100011;   // LI, Version, Mode
  packetBuffer[1] = 0;     // Stratum, or type of clock
  packetBuffer[2] = 6;     // Polling Interval
  packetBuffer[3] = 0xEC;  // Peer Clock Precision
  // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12]  = 49;
  packetBuffer[13]  = 0x4E;
  packetBuffer[14]  = 49;
  packetBuffer[15]  = 52;
  // all NTP fields have been given values, now
  // you can send a packet requesting a timestamp:
  udp.beginPacket(ntpServerIP, 123); //NTP requests are to port 123
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();
}

bool syncClock() {
  if ((millis() > lastSync + syncInterval)     // Update after syncInterval
      || lastSync == 0) {                            // Update if there was no update yet.

    sendNTPPacket();

    // Wait till data is there or timeout...
    byte timeout = 0;
    int cb = 0;
    do {
      delay(10);
      cb = udp.parsePacket();
      if (timeout > 100) {
        return false; // timeout after 1000 ms
      }
      timeout++;
    } while (cb == 0);
    lastSync = millis() - (10 * (timeout + 1)); // Account for delay in reading the time
    udp.read(packetBuffer, NTP_PACKET_SIZE);

    unsigned long highWord = word(packetBuffer[40], packetBuffer[41]);
    unsigned long lowWord = word(packetBuffer[42], packetBuffer[43]);
    // combine the four bytes (two words) into a long integer
    // this is NTP time (seconds since Jan 1 1900):
    unsigned long secsSince1900 = highWord << 16 | lowWord;

    currentEpoc = secsSince1900 - SEVENZYYEARS + ((timeZone + DST) * SECS_PER_HOUR);
    Serial.println("Updated from NTP Server");

    return true;
  }
}

void setupWiFi() {
  Serial.print("Connecting to ");
  Serial.print(ssid);
  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println();
}

void setup() {
  Serial.begin(115200);
  setupWiFi();
  udp.begin(NTP_DEFAULT_LOCAL_PORT);   // listen on this UDP port
}

void loop() {
  syncClock();
  printFormattedTime();
  delay(1000);
}
